/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.shaded.hadoop2.org.apache.curator.framework.recipes.cache;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.flink.shaded.hadoop2.com.google.common.annotations.VisibleForTesting;
import org.apache.flink.shaded.hadoop2.com.google.common.base.Function;
import org.apache.flink.shaded.hadoop2.com.google.common.base.Preconditions;
import org.apache.flink.shaded.hadoop2.com.google.common.collect.ImmutableMap;
import org.apache.flink.shaded.hadoop2.com.google.common.collect.Maps;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.CuratorFramework;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.api.BackgroundCallback;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.api.BackgroundPathable;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.api.CuratorEvent;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.api.GetDataWatchBackgroundStatable;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.api.Pathable;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.api.UnhandledErrorListener;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.listen.Listenable;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.listen.ListenerContainer;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.recipes.cache.TreeCacheEvent;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.recipes.cache.TreeCacheListener;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.state.ConnectionState;
import org.apache.flink.shaded.hadoop2.org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.flink.shaded.hadoop2.org.apache.curator.utils.CloseableExecutorService;
import org.apache.flink.shaded.hadoop2.org.apache.curator.utils.PathUtils;
import org.apache.flink.shaded.hadoop2.org.apache.curator.utils.ThreadUtils;
import org.apache.flink.shaded.hadoop2.org.apache.curator.utils.ZKPaths;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TreeCache
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(TreeCache.class);
    private final AtomicLong outstandingOps = new AtomicLong(0L);
    private final AtomicBoolean isInitialized = new AtomicBoolean(false);
    private final TreeNode root;
    private final CuratorFramework client;
    private final CloseableExecutorService executorService;
    private final boolean cacheData;
    private final boolean dataIsCompressed;
    private final int maxDepth;
    private final ListenerContainer<TreeCacheListener> listeners = new ListenerContainer();
    private final ListenerContainer<UnhandledErrorListener> errorListeners = new ListenerContainer();
    private final AtomicReference<TreeState> treeState = new AtomicReference<TreeState>(TreeState.LATENT);
    private final ConnectionStateListener connectionStateListener = new ConnectionStateListener(){

        @Override
        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            TreeCache.this.handleStateChange(newState);
        }
    };
    static final ThreadFactory defaultThreadFactory = ThreadUtils.newThreadFactory("TreeCache");

    public static Builder newBuilder(CuratorFramework client, String path) {
        return new Builder(client, path);
    }

    public TreeCache(CuratorFramework client, String path) {
        this(client, path, true, false, Integer.MAX_VALUE, new CloseableExecutorService(Executors.newSingleThreadExecutor(defaultThreadFactory), true));
    }

    TreeCache(CuratorFramework client, String path, boolean cacheData, boolean dataIsCompressed, int maxDepth, CloseableExecutorService executorService) {
        this.root = new TreeNode(PathUtils.validatePath(path), null);
        this.client = client;
        this.cacheData = cacheData;
        this.dataIsCompressed = dataIsCompressed;
        this.maxDepth = maxDepth;
        this.executorService = executorService;
    }

    public TreeCache start() throws Exception {
        Preconditions.checkState(this.treeState.compareAndSet(TreeState.LATENT, TreeState.STARTED), "already started");
        this.client.getConnectionStateListenable().addListener(this.connectionStateListener);
        if (this.client.getZookeeperClient().isConnected()) {
            this.root.wasCreated();
        }
        return this;
    }

    @Override
    public void close() {
        if (this.treeState.compareAndSet(TreeState.STARTED, TreeState.CLOSED)) {
            this.client.getConnectionStateListenable().removeListener(this.connectionStateListener);
            this.listeners.clear();
            this.executorService.close();
            try {
                this.root.wasDeleted();
            }
            catch (Exception e) {
                this.handleException(e);
            }
        }
    }

    public Listenable<TreeCacheListener> getListenable() {
        return this.listeners;
    }

    @VisibleForTesting
    Listenable<UnhandledErrorListener> getUnhandledErrorListenable() {
        return this.errorListeners;
    }

    private TreeNode find(String fullPath) {
        if (!fullPath.startsWith(this.root.path)) {
            return null;
        }
        TreeNode current = this.root;
        if (fullPath.length() > this.root.path.length()) {
            if (this.root.path.length() > 1) {
                fullPath = fullPath.substring(this.root.path.length());
            }
            List<String> split = ZKPaths.split(fullPath);
            for (String part : split) {
                ConcurrentMap<String, TreeNode> map = current.children.get();
                if (map == null) {
                    return null;
                }
                current = (TreeNode)map.get(part);
                if (current != null) continue;
                return null;
            }
        }
        return current;
    }

    public Map<String, ChildData> getCurrentChildren(String fullPath) {
        ImmutableMap<String, ChildData> result;
        TreeNode node = this.find(fullPath);
        if (node == null || node.nodeState.get() != NodeState.LIVE) {
            return null;
        }
        ConcurrentMap<String, TreeNode> map = node.children.get();
        if (map == null) {
            result = ImmutableMap.of();
        } else {
            ImmutableMap.Builder builder = ImmutableMap.builder();
            for (Map.Entry entry : map.entrySet()) {
                TreeNode childNode = (TreeNode)entry.getValue();
                ChildData childData = new ChildData(childNode.path, childNode.stat.get(), childNode.data.get());
                if (childNode.nodeState.get() != NodeState.LIVE) continue;
                builder.put(entry.getKey(), childData);
            }
            result = builder.build();
        }
        return node.nodeState.get() == NodeState.LIVE ? result : null;
    }

    public ChildData getCurrentData(String fullPath) {
        TreeNode node = this.find(fullPath);
        if (node == null || node.nodeState.get() != NodeState.LIVE) {
            return null;
        }
        ChildData result = new ChildData(node.path, node.stat.get(), node.data.get());
        return node.nodeState.get() == NodeState.LIVE ? result : null;
    }

    private void callListeners(final TreeCacheEvent event) {
        this.listeners.forEach(new Function<TreeCacheListener, Void>(){

            @Override
            public Void apply(TreeCacheListener listener) {
                try {
                    listener.childEvent(TreeCache.this.client, event);
                }
                catch (Exception e) {
                    TreeCache.this.handleException(e);
                }
                return null;
            }
        });
    }

    private void handleException(final Throwable e) {
        if (this.errorListeners.size() == 0) {
            LOG.error("", e);
        } else {
            this.errorListeners.forEach(new Function<UnhandledErrorListener, Void>(){

                @Override
                public Void apply(UnhandledErrorListener listener) {
                    try {
                        listener.unhandledError("", e);
                    }
                    catch (Exception e2) {
                        LOG.error("Exception handling exception", (Throwable)e2);
                    }
                    return null;
                }
            });
        }
    }

    private void handleStateChange(ConnectionState newState) {
        switch (newState) {
            case SUSPENDED: {
                this.publishEvent(TreeCacheEvent.Type.CONNECTION_SUSPENDED);
                break;
            }
            case LOST: {
                this.publishEvent(TreeCacheEvent.Type.CONNECTION_LOST);
                break;
            }
            case CONNECTED: {
                try {
                    this.root.wasCreated();
                }
                catch (Exception e) {
                    this.handleException(e);
                }
                break;
            }
            case RECONNECTED: {
                try {
                    this.root.wasReconnected();
                    this.publishEvent(TreeCacheEvent.Type.CONNECTION_RECONNECTED);
                    break;
                }
                catch (Exception e) {
                    this.handleException(e);
                }
            }
        }
    }

    private void publishEvent(TreeCacheEvent.Type type) {
        this.publishEvent(new TreeCacheEvent(type, null));
    }

    private void publishEvent(TreeCacheEvent.Type type, String path) {
        this.publishEvent(new TreeCacheEvent(type, new ChildData(path, null, null)));
    }

    private void publishEvent(TreeCacheEvent.Type type, ChildData data) {
        this.publishEvent(new TreeCacheEvent(type, data));
    }

    private void publishEvent(final TreeCacheEvent event) {
        if (this.treeState.get() != TreeState.CLOSED) {
            this.executorService.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        TreeCache.this.callListeners(event);
                    }
                    catch (Exception e) {
                        TreeCache.this.handleException(e);
                    }
                }
            });
        }
    }

    private static enum TreeState {
        LATENT,
        STARTED,
        CLOSED;

    }

    private final class TreeNode
    implements Watcher,
    BackgroundCallback {
        final AtomicReference<NodeState> nodeState = new AtomicReference<NodeState>(NodeState.PENDING);
        final TreeNode parent;
        final String path;
        final AtomicReference<Stat> stat = new AtomicReference();
        final AtomicReference<byte[]> data = new AtomicReference();
        final AtomicReference<ConcurrentMap<String, TreeNode>> children = new AtomicReference();
        final int depth;

        TreeNode(String path, TreeNode parent) {
            this.path = path;
            this.parent = parent;
            this.depth = parent == null ? 0 : parent.depth + 1;
        }

        private void refresh() throws Exception {
            if (this.depth < TreeCache.this.maxDepth) {
                TreeCache.this.outstandingOps.addAndGet(2L);
                this.doRefreshData();
                this.doRefreshChildren();
            } else {
                this.refreshData();
            }
        }

        private void refreshChildren() throws Exception {
            if (this.depth < TreeCache.this.maxDepth) {
                TreeCache.this.outstandingOps.incrementAndGet();
                this.doRefreshChildren();
            }
        }

        private void refreshData() throws Exception {
            TreeCache.this.outstandingOps.incrementAndGet();
            this.doRefreshData();
        }

        private void doRefreshChildren() throws Exception {
            ((Pathable)((BackgroundPathable)TreeCache.this.client.getChildren().usingWatcher(this)).inBackground(this)).forPath(this.path);
        }

        private void doRefreshData() throws Exception {
            if (TreeCache.this.dataIsCompressed) {
                ((Pathable)((BackgroundPathable)((GetDataWatchBackgroundStatable)TreeCache.this.client.getData().decompressed()).usingWatcher(this)).inBackground(this)).forPath(this.path);
            } else {
                ((Pathable)((BackgroundPathable)TreeCache.this.client.getData().usingWatcher(this)).inBackground(this)).forPath(this.path);
            }
        }

        void wasReconnected() throws Exception {
            this.refresh();
            ConcurrentMap<String, TreeNode> childMap = this.children.get();
            if (childMap != null) {
                for (TreeNode child : childMap.values()) {
                    child.wasReconnected();
                }
            }
        }

        void wasCreated() throws Exception {
            this.refresh();
        }

        void wasDeleted() throws Exception {
            this.stat.set(null);
            this.data.set(null);
            TreeCache.this.client.clearWatcherReferences(this);
            ConcurrentMap childMap = this.children.getAndSet(null);
            if (childMap != null) {
                ArrayList childCopy = new ArrayList(childMap.values());
                childMap.clear();
                for (TreeNode child : childCopy) {
                    child.wasDeleted();
                }
            }
            if (TreeCache.this.treeState.get() == TreeState.CLOSED) {
                return;
            }
            if (this.nodeState.compareAndSet(NodeState.LIVE, NodeState.DEAD)) {
                TreeCache.this.publishEvent(TreeCacheEvent.Type.NODE_REMOVED, this.path);
            }
            if (this.parent == null) {
                ((Pathable)((BackgroundPathable)TreeCache.this.client.checkExists().usingWatcher(this)).inBackground()).forPath(this.path);
            } else {
                ConcurrentMap<String, TreeNode> parentChildMap = this.parent.children.get();
                if (parentChildMap != null) {
                    parentChildMap.remove(ZKPaths.getNodeFromPath(this.path), this);
                }
            }
        }

        @Override
        public void process(WatchedEvent event) {
            try {
                switch (event.getType()) {
                    case NodeCreated: {
                        Preconditions.checkState(this.parent == null, "unexpected NodeCreated on non-root node");
                        this.wasCreated();
                        break;
                    }
                    case NodeChildrenChanged: {
                        this.refreshChildren();
                        break;
                    }
                    case NodeDataChanged: {
                        this.refreshData();
                        break;
                    }
                    case NodeDeleted: {
                        this.wasDeleted();
                    }
                }
            }
            catch (Exception e) {
                TreeCache.this.handleException(e);
            }
        }

        @Override
        public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
            Stat newStat = event.getStat();
            switch (event.getType()) {
                case EXISTS: {
                    Preconditions.checkState(this.parent == null, "unexpected EXISTS on non-root node");
                    if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                        this.nodeState.compareAndSet(NodeState.DEAD, NodeState.PENDING);
                        this.wasCreated();
                        break;
                    }
                    if (event.getResultCode() != KeeperException.Code.NONODE.intValue()) break;
                    this.wasDeleted();
                    break;
                }
                case CHILDREN: {
                    if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                        Stat oldStat = this.stat.get();
                        if (oldStat != null && oldStat.getMzxid() == newStat.getMzxid()) {
                            this.stat.set(newStat);
                        }
                        if (event.getChildren().isEmpty()) break;
                        ConcurrentMap<String, TreeNode> childMap = this.children.get();
                        if (childMap == null && !this.children.compareAndSet(null, childMap = Maps.newConcurrentMap())) {
                            childMap = this.children.get();
                        }
                        ArrayList<String> newChildren = new ArrayList<String>();
                        for (String child : event.getChildren()) {
                            if (childMap.containsKey(child)) continue;
                            newChildren.add(child);
                        }
                        Collections.sort(newChildren);
                        for (String child : newChildren) {
                            String fullPath;
                            TreeNode node;
                            if (childMap.putIfAbsent(child, node = new TreeNode(fullPath = ZKPaths.makePath(this.path, child), this)) != null) continue;
                            node.wasCreated();
                        }
                        break;
                    }
                    if (event.getResultCode() != KeeperException.Code.NONODE.intValue()) break;
                    this.wasDeleted();
                    break;
                }
                case GET_DATA: {
                    if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                        if (TreeCache.this.cacheData) {
                            this.data.set(event.getData());
                        }
                        Stat oldStat = this.stat.getAndSet(newStat);
                        if (this.nodeState.compareAndSet(NodeState.PENDING, NodeState.LIVE)) {
                            TreeCache.this.publishEvent(TreeCacheEvent.Type.NODE_ADDED, new ChildData(event.getPath(), newStat, event.getData()));
                            break;
                        }
                        if (oldStat != null && oldStat.getMzxid() == newStat.getMzxid()) break;
                        TreeCache.this.publishEvent(TreeCacheEvent.Type.NODE_UPDATED, new ChildData(event.getPath(), newStat, event.getData()));
                        break;
                    }
                    if (event.getResultCode() != KeeperException.Code.NONODE.intValue()) break;
                    this.wasDeleted();
                    break;
                }
                default: {
                    LOG.info(String.format("Unknown event %s", event));
                    TreeCache.this.outstandingOps.decrementAndGet();
                    return;
                }
            }
            if (TreeCache.this.outstandingOps.decrementAndGet() == 0L && TreeCache.this.isInitialized.compareAndSet(false, true)) {
                TreeCache.this.publishEvent(TreeCacheEvent.Type.INITIALIZED);
            }
        }
    }

    private static enum NodeState {
        PENDING,
        LIVE,
        DEAD;

    }

    public static final class Builder {
        private final CuratorFramework client;
        private final String path;
        private boolean cacheData = true;
        private boolean dataIsCompressed = false;
        private CloseableExecutorService executorService = null;
        private int maxDepth = Integer.MAX_VALUE;

        private Builder(CuratorFramework client, String path) {
            this.client = Preconditions.checkNotNull(client);
            this.path = PathUtils.validatePath(path);
        }

        public TreeCache build() {
            CloseableExecutorService executor = this.executorService;
            if (executor == null) {
                executor = new CloseableExecutorService(Executors.newSingleThreadExecutor(defaultThreadFactory));
            }
            return new TreeCache(this.client, this.path, this.cacheData, this.dataIsCompressed, this.maxDepth, executor);
        }

        public Builder setCacheData(boolean cacheData) {
            this.cacheData = cacheData;
            return this;
        }

        public Builder setDataIsCompressed(boolean dataIsCompressed) {
            this.dataIsCompressed = dataIsCompressed;
            return this;
        }

        public Builder setExecutor(ThreadFactory threadFactory) {
            return this.setExecutor(new CloseableExecutorService(Executors.newSingleThreadExecutor(threadFactory)));
        }

        public Builder setExecutor(ExecutorService executorService) {
            if (executorService instanceof CloseableExecutorService) {
                return this.setExecutor((CloseableExecutorService)((Object)executorService));
            }
            return this.setExecutor(new CloseableExecutorService(executorService));
        }

        public Builder setExecutor(CloseableExecutorService executorService) {
            this.executorService = Preconditions.checkNotNull(executorService);
            return this;
        }

        public Builder setMaxDepth(int maxDepth) {
            this.maxDepth = maxDepth;
            return this;
        }
    }
}

