/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.logredactor.internal;

import com.eclipsesource.json.Json;
import com.eclipsesource.json.JsonArray;
import com.eclipsesource.json.JsonObject;
import com.eclipsesource.json.JsonValue;
import com.eclipsesource.json.ParseException;
import com.google.re2j.Matcher;
import com.google.re2j.Pattern;
import io.confluent.log4j.redactor.LogRedactorMetrics;
import io.confluent.logredactor.internal.MetricsTagBuilder;
import io.confluent.logredactor.internal.RedactionPolicyParseException;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;

public class StringRedactor {
    private RedactionPolicy policy;

    public static StringRedactor emptyStringRedactor(LogRedactorMetrics metrics, Map<String, String> tags) {
        StringRedactor sr = new StringRedactor();
        sr.policy = RedactionPolicy.emptyRedactionPolicy(metrics, tags);
        return sr;
    }

    private StringRedactor() {
    }

    public static StringRedactor createFromJsonFile(String fileName) throws IOException {
        return StringRedactor.createFromJsonFile(fileName, LogRedactorMetrics.NOOP);
    }

    public static StringRedactor createFromJsonFile(String fileName, LogRedactorMetrics metrics) throws IOException {
        StringRedactor sr = new StringRedactor();
        if (fileName == null) {
            sr.policy = RedactionPolicy.emptyRedactionPolicy(metrics, new MetricsTagBuilder().policyLocation("").policyHash("").build());
            return sr;
        }
        File file = new File(fileName);
        if (file.exists() && file.length() == 0L) {
            sr.policy = RedactionPolicy.emptyRedactionPolicy(metrics, new MetricsTagBuilder().policyLocation(fileName).policyHash("").build());
            return sr;
        }
        String content = new String(Files.readAllBytes(Paths.get(fileName, new String[0])));
        return StringRedactor.createFromJsonString(content, fileName, metrics);
    }

    public static StringRedactor createFromJsonString(String json, String rulesLocation) throws IOException {
        return StringRedactor.createFromJsonString(json, rulesLocation, LogRedactorMetrics.NOOP);
    }

    public static StringRedactor createFromJsonString(String json, String rulesLocation, LogRedactorMetrics metrics) throws IOException {
        RedactionPolicy policy;
        StringRedactor sr = new StringRedactor();
        if (json == null || json.isEmpty() || rulesLocation == null || rulesLocation.isEmpty()) {
            sr.policy = RedactionPolicy.emptyRedactionPolicy(metrics, new MetricsTagBuilder().policyLocation(rulesLocation).policyContent(json).build());
            return sr;
        }
        try {
            policy = StringRedactor.createPolicyFromJson(json);
        }
        catch (RedactionPolicyParseException | RuntimeException e) {
            throw new RedactionPolicyParseException(e.getMessage(), e.getCause());
        }
        policy.postProcess();
        sr.policy = policy;
        sr.policy.setupMetrics(metrics, new MetricsTagBuilder().policyLocation(rulesLocation).policyContent(json).build());
        return sr;
    }

    private static RedactionPolicy createPolicyFromJson(String json) throws RedactionPolicyParseException, ParseException {
        RedactionPolicy policy = new RedactionPolicy();
        JsonObject jo = Json.parse((String)json).asObject();
        if (jo.size() > 2) {
            throw new RedactionPolicyParseException("Too many fields");
        }
        JsonValue version = jo.get("version");
        if (version != null) {
            int policyVersion;
            if (version.isString()) {
                String strVersion = version.asString();
                policyVersion = Integer.parseInt(strVersion);
            } else if (version.isNumber()) {
                policyVersion = jo.getInt("version", -1);
            } else {
                throw new RedactionPolicyParseException("Not a number");
            }
            policy.setVersion(policyVersion);
        }
        JsonArray rules = jo.get("rules").asArray();
        ArrayList<RedactionRule> policyRules = new ArrayList<RedactionRule>();
        for (JsonValue rule : rules) {
            RedactionRule newRule = new RedactionRule();
            String description = rule.asObject().getString("description", null);
            newRule.setDescription(description);
            boolean caseSensitive = StringRedactor.parseBoolean(rule, "caseSensitive", true);
            newRule.setCaseSensitive(caseSensitive);
            boolean omitNestedExceptions = StringRedactor.parseBoolean(rule, "omitNestedExceptions", false);
            newRule.setOmitNestedExceptions(omitNestedExceptions);
            String trigger = rule.asObject().getString("trigger", null);
            newRule.setTrigger(trigger);
            String search = rule.asObject().getString("search", null);
            newRule.setSearch(search);
            String replace = rule.asObject().getString("replace", null);
            newRule.setReplace(replace);
            String metricsId = rule.asObject().getString("metricsId", null);
            newRule.setMetricsId(metricsId);
            System.out.println(rule);
            policyRules.add(newRule);
        }
        policy.setRules(policyRules);
        return policy;
    }

    private static boolean parseBoolean(JsonValue rule, String name, boolean defaultValue) throws RedactionPolicyParseException {
        boolean value = defaultValue;
        JsonValue policyBooleanValue = rule.asObject().get(name);
        if (policyBooleanValue != null) {
            if (policyBooleanValue.isString()) {
                String strValue = policyBooleanValue.asString();
                value = Boolean.parseBoolean(strValue);
            } else if (policyBooleanValue.isBoolean()) {
                value = rule.asObject().getBoolean(name, defaultValue);
            } else {
                throw new RedactionPolicyParseException("Not a boolean");
            }
        }
        return value;
    }

    public String redact(String msg) {
        return this.policy.redact(msg);
    }

    public Throwable redact(Throwable t) {
        if (t != null) {
            Throwable redactedCause;
            String msg = t.getMessage();
            AtomicBoolean omitNestedExceptions = new AtomicBoolean(false);
            String redactedMsg = this.policy.redact(msg, rule -> {
                if (((RedactionRule)rule).omitNestedExceptions) {
                    omitNestedExceptions.set(true);
                }
                return null;
            });
            Throwable cause = t.getCause();
            Throwable throwable = redactedCause = omitNestedExceptions.get() ? null : this.redact(cause);
            if (redactedMsg != msg || redactedCause != cause) {
                Throwable redacted;
                try {
                    redacted = (Throwable)t.getClass().getDeclaredConstructor(String.class, Throwable.class).newInstance(redactedMsg, redactedCause);
                }
                catch (IllegalAccessException | InstantiationException | NoSuchMethodException | RuntimeException | InvocationTargetException e) {
                    redacted = new Throwable(redactedMsg, redactedCause);
                }
                redacted.setStackTrace(t.getStackTrace());
                return redacted;
            }
        }
        return t;
    }

    public void measureRuleCount() {
        this.policy.measureRuleCount();
    }

    private static class RedactionPolicy {
        private int version = -1;
        private List<RedactionRule> rules;
        public LogRedactorMetrics metrics;
        public Map<String, String> tags;

        private RedactionPolicy() {
        }

        public void setVersion(int version) {
            this.version = version;
        }

        public void setRules(List<RedactionRule> rules) {
            this.rules = rules;
        }

        public void setupMetrics(LogRedactorMetrics metrics, Map<String, String> tags) {
            this.metrics = metrics;
            this.tags = tags;
            HashMap<String, String> ruleTags = new HashMap<String, String>(tags);
            for (RedactionRule rule : this.rules) {
                if (rule.metricsId == null) continue;
                ruleTags.put("rule", rule.metricsId);
                rule.setTags(ruleTags);
            }
            metrics.count("policy_update", tags);
            this.measureRuleCount();
        }

        private static RedactionPolicy emptyRedactionPolicy(LogRedactorMetrics metrics, Map<String, String> tags) {
            RedactionPolicy policy = new RedactionPolicy();
            policy.version = 1;
            policy.rules = new ArrayList<RedactionRule>();
            policy.setupMetrics(metrics, tags);
            return policy;
        }

        private void postProcess() throws RedactionPolicyParseException {
            if (this.version == -1) {
                throw new RedactionPolicyParseException("No version specified.");
            }
            if (this.version != 1) {
                throw new RedactionPolicyParseException("Unknown version " + this.version);
            }
            for (RedactionRule rule : this.rules) {
                rule.postProcess();
            }
        }

        private String redact(String msg) {
            return this.redact(msg, null);
        }

        private String redact(String msg, Function<RedactionRule, Void> matchingRuleCallback) {
            if (msg == null) {
                return null;
            }
            String original = msg;
            boolean matched = false;
            boolean redacted = false;
            for (RedactionRule rule : this.rules) {
                if (!rule.matchesTrigger(msg)) continue;
                Matcher m = (Matcher)rule.matcherTL.get();
                m.reset((CharSequence)msg);
                if (!m.find()) continue;
                matched = true;
                if (rule.metricsId != null) {
                    this.metrics.count("matches", rule.tags);
                }
                if (rule.replace != null) {
                    msg = m.replaceAll(rule.replace);
                    redacted = true;
                    if (rule.metricsId != null) {
                        this.metrics.count("redactions", rule.tags);
                    }
                }
                if (matchingRuleCallback == null) continue;
                matchingRuleCallback.apply(rule);
            }
            this.metrics.count("scanned_log_statements", this.tags);
            if (matched) {
                this.metrics.count("matched_log_statements", this.tags);
            }
            if (redacted) {
                this.metrics.count("redacted_log_statements", this.tags);
                return msg;
            }
            return original;
        }

        public void measureRuleCount() {
            this.metrics.gauge((double)this.rules.size(), "policy_rule_count", this.tags);
        }
    }

    private static class RedactionRule {
        private String description;
        private boolean caseSensitive = true;
        private boolean omitNestedExceptions = false;
        private String trigger;
        private String search;
        private String replace;
        private Pattern pattern;
        private ThreadLocal<Matcher> matcherTL;
        private String metricsId;
        private Map<String, String> tags = new HashMap<String, String>();

        private RedactionRule() {
        }

        public void setDescription(String description) {
            this.description = description;
        }

        public void setCaseSensitive(boolean caseSensitive) {
            this.caseSensitive = caseSensitive;
        }

        public void setOmitNestedExceptions(boolean omitNestedExceptions) {
            this.omitNestedExceptions = omitNestedExceptions;
        }

        public void setTrigger(String trigger) {
            this.trigger = trigger;
        }

        public void setMetricsId(String metricsId) {
            this.metricsId = metricsId;
            this.tags.put("rule", metricsId);
        }

        public void setTags(Map<String, String> tags) {
            this.tags = new HashMap<String, String>(tags);
            this.tags.put("rule", this.metricsId);
        }

        public void setSearch(String search) {
            this.search = search;
            if (search != null) {
                Pattern pattern = Pattern.compile((String)search);
            }
        }

        public void setReplace(String replace) {
            this.replace = replace;
        }

        private void validateReplacement(Pattern pattern, String replacement) {
            int i = 0;
            int m = replacement.length();
            int groupCount = pattern.groupCount();
            Set<Object> namedGroups = Collections.emptySet();
            try {
                namedGroups = pattern.namedGroups().keySet();
            }
            catch (NullPointerException nullPointerException) {
                // empty catch block
            }
            while (i < m - 1) {
                if (replacement.charAt(i) == '\\') {
                    ++i;
                } else if (replacement.charAt(i) == '$') {
                    char c = replacement.charAt(i + 1);
                    if ('0' <= c && c <= '9') {
                        int n = c - 48;
                        i += 2;
                        while (i < m && (c = replacement.charAt(i)) >= '0' && c <= '9' && n * 10 + c - 48 <= groupCount) {
                            n = n * 10 + c - 48;
                            ++i;
                        }
                        if (n > groupCount) {
                            throw new IndexOutOfBoundsException("Replacement string contains a group number '" + n + "' which is > total number of groups");
                        }
                        --i;
                    } else if (c == '{') {
                        int j;
                        for (j = ++i + 1; j < replacement.length() && replacement.charAt(j) != '}' && replacement.charAt(j) != ' '; ++j) {
                        }
                        if (j == replacement.length() || replacement.charAt(j) != '}') {
                            throw new IllegalArgumentException("In replacement string, named capture group is missing trailing '}'");
                        }
                        String groupName = replacement.substring(i + 1, j);
                        if (!namedGroups.contains(groupName)) {
                            throw new IllegalArgumentException("Replacement string contains a group '" + groupName + "' which is not found in the pattern");
                        }
                    }
                }
                ++i;
            }
        }

        private void postProcess() throws RedactionPolicyParseException {
            if (this.search == null || this.search.isEmpty()) {
                throw new RedactionPolicyParseException("The search regular expression cannot be empty.");
            }
            this.pattern = this.caseSensitive ? Pattern.compile((String)this.search) : Pattern.compile((String)this.search, (int)1);
            this.matcherTL = ThreadLocal.withInitial(() -> this.pattern.matcher((CharSequence)""));
            try {
                String sampleString = "Hello, world";
                Matcher m = this.pattern.matcher((CharSequence)sampleString);
                if (this.replace != null) {
                    this.validateReplacement(this.pattern, this.replace);
                    sampleString = m.replaceAll(this.replace);
                }
            }
            catch (Exception e) {
                throw new RedactionPolicyParseException("The replacement text \"" + this.replace + "\" is invalid: " + e.getMessage(), e);
            }
        }

        private boolean matchesTrigger(String msg) {
            if (this.trigger == null || this.trigger.isEmpty()) {
                return true;
            }
            if (this.caseSensitive) {
                return msg.contains(this.trigger);
            }
            int len = this.trigger.length();
            int max = msg.length() - len;
            for (int i = 0; i <= max; ++i) {
                if (!msg.regionMatches(true, i, this.trigger, 0, len)) continue;
                return true;
            }
            return false;
        }
    }
}

