/*
 * Decompiled with CFR 0.152.
 */
package org.apache.twill.discovery;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.hash.Hashing;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.twill.common.Cancellable;
import org.apache.twill.common.Threads;
import org.apache.twill.discovery.DefaultServiceDiscovered;
import org.apache.twill.discovery.Discoverable;
import org.apache.twill.discovery.DiscoverableAdapter;
import org.apache.twill.discovery.DiscoverableWrapper;
import org.apache.twill.discovery.DiscoveryService;
import org.apache.twill.discovery.DiscoveryServiceClient;
import org.apache.twill.discovery.ServiceDiscovered;
import org.apache.twill.zookeeper.NodeChildren;
import org.apache.twill.zookeeper.NodeData;
import org.apache.twill.zookeeper.OperationFuture;
import org.apache.twill.zookeeper.ZKClient;
import org.apache.twill.zookeeper.ZKClients;
import org.apache.twill.zookeeper.ZKOperations;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKDiscoveryService
implements DiscoveryService,
DiscoveryServiceClient {
    private static final Logger LOG = LoggerFactory.getLogger(ZKDiscoveryService.class);
    private static final String NAMESPACE = "/discoverable";
    private static final long RETRY_MILLIS = 1000L;
    private final Multimap<Discoverable, DiscoveryCancellable> discoverables = HashMultimap.create();
    private final Lock lock = new ReentrantLock();
    private final LoadingCache<String, ServiceDiscovered> services;
    private final ZKClient zkClient;
    private final ScheduledExecutorService retryExecutor = Executors.newSingleThreadScheduledExecutor(Threads.createDaemonThreadFactory((String)"zk-discovery-retry"));

    public ZKDiscoveryService(ZKClient zkClient) {
        this(zkClient, NAMESPACE);
    }

    public ZKDiscoveryService(ZKClient zkClient, String namespace) {
        this.zkClient = namespace == null ? zkClient : ZKClients.namespace((ZKClient)zkClient, (String)namespace);
        this.services = CacheBuilder.newBuilder().build(this.createServiceLoader());
        this.zkClient.addConnectionWatcher(this.createConnectionWatcher());
    }

    public Cancellable register(final Discoverable discoverable) {
        final DiscoverableWrapper wrapper = new DiscoverableWrapper(discoverable);
        final SettableFuture future = SettableFuture.create();
        final DiscoveryCancellable cancellable = new DiscoveryCancellable(wrapper);
        Futures.addCallback(this.doRegister(wrapper), (FutureCallback)new FutureCallback<String>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onSuccess(String result) {
                cancellable.setPath(result);
                ZKDiscoveryService.this.lock.lock();
                try {
                    ZKDiscoveryService.this.discoverables.put((Object)wrapper, (Object)cancellable);
                }
                finally {
                    ZKDiscoveryService.this.lock.unlock();
                }
                LOG.debug("Service registered: {} {}", (Object)wrapper, (Object)result);
                future.set((Object)result);
            }

            public void onFailure(Throwable t) {
                if (t instanceof KeeperException.NodeExistsException) {
                    ZKDiscoveryService.this.handleRegisterFailure(discoverable, (SettableFuture<String>)future, (FutureCallback<String>)this, t);
                } else {
                    LOG.warn("Failed to register: {}", (Object)wrapper, (Object)t);
                    future.setException(t);
                }
            }
        }, (Executor)Threads.SAME_THREAD_EXECUTOR);
        Futures.getUnchecked((Future)future);
        return cancellable;
    }

    public ServiceDiscovered discover(String service) {
        return (ServiceDiscovered)this.services.getUnchecked((Object)service);
    }

    private void handleRegisterFailure(final Discoverable discoverable, final SettableFuture<String> completion, final FutureCallback<String> creationCallback, final Throwable failureCause) {
        final String path = this.getNodePath(discoverable);
        Futures.addCallback((ListenableFuture)this.zkClient.exists(path), (FutureCallback)new FutureCallback<Stat>(){

            public void onSuccess(Stat result) {
                if (result == null) {
                    LOG.info("Node {} is gone. Retry registration for {}.", (Object)path, (Object)discoverable);
                    ZKDiscoveryService.this.retryRegister(discoverable, (FutureCallback<String>)creationCallback);
                    return;
                }
                long ephemeralOwner = result.getEphemeralOwner();
                if (ephemeralOwner == 0L) {
                    LOG.error("Node {} already exists and is not an ephemeral node. Discoverable registration failed: {}.", (Object)path, (Object)discoverable);
                    completion.setException(failureCause);
                    return;
                }
                Long sessionId = ZKDiscoveryService.this.zkClient.getSessionId();
                if (sessionId == null || ephemeralOwner != sessionId) {
                    LOG.info("Owner of {} is different. Retry registration for {}.", (Object)path, (Object)discoverable);
                    ZKDiscoveryService.this.retryRegister(discoverable, (FutureCallback<String>)creationCallback);
                } else {
                    completion.set((Object)path);
                }
            }

            public void onFailure(Throwable t) {
                LOG.warn("Error when getting stats on {}. Retry registration for {}.", (Object)path, (Object)discoverable);
                ZKDiscoveryService.this.retryRegister(discoverable, (FutureCallback<String>)creationCallback);
            }
        }, (Executor)Threads.SAME_THREAD_EXECUTOR);
    }

    private OperationFuture<String> doRegister(Discoverable discoverable) {
        byte[] discoverableBytes = DiscoverableAdapter.encode(discoverable);
        return this.zkClient.create(this.getNodePath(discoverable), discoverableBytes, CreateMode.EPHEMERAL, true);
    }

    private void retryRegister(final Discoverable discoverable, final FutureCallback<String> creationCallback) {
        this.retryExecutor.schedule(new Runnable(){

            @Override
            public void run() {
                Futures.addCallback((ListenableFuture)ZKDiscoveryService.this.doRegister(discoverable), (FutureCallback)creationCallback, (Executor)Threads.SAME_THREAD_EXECUTOR);
            }
        }, 1000L, TimeUnit.MILLISECONDS);
    }

    private String getNodePath(Discoverable discoverable) {
        InetSocketAddress socketAddress = discoverable.getSocketAddress();
        String node = Hashing.md5().newHasher().putBytes(socketAddress.getAddress().getAddress()).putInt(socketAddress.getPort()).hash().toString();
        return String.format("/%s/%s", discoverable.getName(), node);
    }

    private Watcher createConnectionWatcher() {
        return new Watcher(){
            private boolean expired;

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void process(WatchedEvent event) {
                if (event.getState() == Watcher.Event.KeeperState.Expired) {
                    LOG.warn("ZK Session expired: {}", (Object)ZKDiscoveryService.this.zkClient.getConnectString());
                    this.expired = true;
                } else if (event.getState() == Watcher.Event.KeeperState.SyncConnected && this.expired) {
                    LOG.info("Reconnected after expiration: {}", (Object)ZKDiscoveryService.this.zkClient.getConnectString());
                    this.expired = false;
                    ZKDiscoveryService.this.lock.lock();
                    try {
                        for (final Map.Entry entry : ZKDiscoveryService.this.discoverables.entries()) {
                            LOG.info("Re-registering service: {}", entry.getKey());
                            Futures.addCallback((ListenableFuture)ZKDiscoveryService.this.doRegister((Discoverable)entry.getKey()), (FutureCallback)new FutureCallback<String>(){

                                public void onSuccess(String result) {
                                    ((DiscoveryCancellable)entry.getValue()).setPath(result);
                                    LOG.debug("Service re-registered: {} {}", entry.getKey(), (Object)result);
                                }

                                public void onFailure(Throwable t) {
                                    ((DiscoveryCancellable)entry.getValue()).setPath(null);
                                    LOG.error("Failed to re-register service: {}", entry.getKey(), (Object)t);
                                }
                            }, (Executor)Threads.SAME_THREAD_EXECUTOR);
                        }
                    }
                    finally {
                        ZKDiscoveryService.this.lock.unlock();
                    }
                }
            }
        };
    }

    private CacheLoader<String, ServiceDiscovered> createServiceLoader() {
        return new CacheLoader<String, ServiceDiscovered>(){

            public ServiceDiscovered load(String service) throws Exception {
                final DefaultServiceDiscovered serviceDiscovered = new DefaultServiceDiscovered(service);
                final String serviceBase = "/" + service;
                ZKOperations.watchChildren((ZKClient)ZKDiscoveryService.this.zkClient, (String)serviceBase, (ZKOperations.ChildrenCallback)new ZKOperations.ChildrenCallback(){

                    public void updated(NodeChildren nodeChildren) {
                        List children = nodeChildren.getChildren();
                        ArrayList dataFutures = Lists.newArrayListWithCapacity((int)children.size());
                        for (String child : children) {
                            dataFutures.add(ZKDiscoveryService.this.zkClient.getData(serviceBase + "/" + child));
                        }
                        final ListenableFuture fetchFuture = Futures.successfulAsList((Iterable)dataFutures);
                        fetchFuture.addListener(new Runnable(){

                            @Override
                            public void run() {
                                ImmutableSet.Builder builder = ImmutableSet.builder();
                                for (NodeData nodeData : (List)Futures.getUnchecked((Future)fetchFuture)) {
                                    Discoverable discoverable;
                                    if (nodeData == null || (discoverable = DiscoverableAdapter.decode(nodeData.getData())) == null) continue;
                                    builder.add((Object)discoverable);
                                }
                                serviceDiscovered.setDiscoverables((Set<Discoverable>)builder.build());
                            }
                        }, Threads.SAME_THREAD_EXECUTOR);
                    }
                });
                return serviceDiscovered;
            }
        };
    }

    private final class DiscoveryCancellable
    implements Cancellable {
        private final Discoverable discoverable;
        private final AtomicBoolean cancelled;
        private volatile String path;

        DiscoveryCancellable(Discoverable discoverable) {
            this.discoverable = discoverable;
            this.cancelled = new AtomicBoolean();
        }

        void setPath(String path) {
            this.path = path;
            if (this.cancelled.get() && path != null) {
                ZKDiscoveryService.this.zkClient.delete(path);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void cancel() {
            if (!this.cancelled.compareAndSet(false, true)) {
                return;
            }
            String path = this.path;
            if (path == null) {
                return;
            }
            ZKDiscoveryService.this.lock.lock();
            try {
                ZKDiscoveryService.this.discoverables.remove((Object)this.discoverable, (Object)this);
            }
            finally {
                ZKDiscoveryService.this.lock.unlock();
            }
            Futures.getUnchecked((Future)ZKOperations.ignoreError((OperationFuture)ZKDiscoveryService.this.zkClient.delete(path), KeeperException.NoNodeException.class, (Object)path));
            LOG.debug("Service unregistered: {} {}", (Object)this.discoverable, (Object)path);
        }
    }
}

