/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Ordering;
import com.google.inject.Inject;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.druid.common.guava.SettableSupplier;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.Intervals;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.emitter.EmittingLogger;
import org.apache.druid.query.DataSegmentAndDescriptor;
import org.apache.druid.query.LeafSegmentsBundle;
import org.apache.druid.query.SegmentDescriptor;
import org.apache.druid.query.TableDataSource;
import org.apache.druid.segment.PhysicalSegmentInspector;
import org.apache.druid.segment.Segment;
import org.apache.druid.segment.SegmentLazyLoadFailCallback;
import org.apache.druid.segment.SegmentMapFunction;
import org.apache.druid.segment.SegmentReference;
import org.apache.druid.segment.join.table.IndexedTable;
import org.apache.druid.segment.join.table.ReferenceCountedIndexedTableProvider;
import org.apache.druid.segment.loading.AcquireSegmentAction;
import org.apache.druid.segment.loading.SegmentCacheManager;
import org.apache.druid.segment.loading.SegmentLoadingException;
import org.apache.druid.server.metrics.SegmentRowCountDistribution;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.SegmentId;
import org.apache.druid.timeline.VersionedIntervalTimeline;
import org.apache.druid.timeline.partition.PartitionChunk;
import org.apache.druid.utils.CloseableUtils;
import org.apache.druid.utils.CollectionUtils;

public class SegmentManager {
    private static final EmittingLogger log = new EmittingLogger(SegmentManager.class);
    private final SegmentCacheManager cacheManager;
    private final ConcurrentHashMap<String, DataSourceState> dataSources = new ConcurrentHashMap();

    @Inject
    public SegmentManager(SegmentCacheManager cacheManager) {
        this.cacheManager = cacheManager;
    }

    @VisibleForTesting
    Map<String, DataSourceState> getDataSources() {
        return this.dataSources;
    }

    public Set<String> getDataSourceNames() {
        return this.dataSources.keySet();
    }

    public Map<String, Long> getDataSourceSizes() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getTotalSegmentSize);
    }

    public Map<String, Long> getAverageRowCountForDatasource() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getAverageRowCount);
    }

    public Map<String, SegmentRowCountDistribution> getRowCountDistribution() {
        return CollectionUtils.mapValues(this.dataSources, rec$ -> ((DataSourceState)rec$).getSegmentRowCountDistribution());
    }

    public Map<String, Long> getDataSourceCounts() {
        return CollectionUtils.mapValues(this.dataSources, DataSourceState::getNumSegments);
    }

    public Optional<VersionedIntervalTimeline<String, DataSegment>> getTimeline(TableDataSource dataSource) {
        return Optional.ofNullable(this.dataSources.get(dataSource.getName())).map(DataSourceState::getTimeline);
    }

    public LeafSegmentsBundle getSegmentsBundle(List<DataSegmentAndDescriptor> segments, SegmentMapFunction segmentMapFunction) {
        ArrayList<SegmentReference> segmentReferences = new ArrayList<SegmentReference>();
        ArrayList<SegmentDescriptor> missingSegments = new ArrayList<SegmentDescriptor>();
        ArrayList<DataSegmentAndDescriptor> loadableSegments = new ArrayList<DataSegmentAndDescriptor>();
        for (DataSegmentAndDescriptor segment : segments) {
            DataSegment dataSegment = segment.getDataSegment();
            if (dataSegment == null) {
                missingSegments.add(segment.getDescriptor());
                continue;
            }
            Optional<Segment> ref = this.acquireCachedSegment(dataSegment);
            if (ref.isPresent()) {
                segmentReferences.add(new SegmentReference(segment.getDescriptor(), (Optional)segmentMapFunction.apply(ref), null));
                continue;
            }
            if (this.canLoadSegmentOnDemand(dataSegment)) {
                loadableSegments.add(segment);
                continue;
            }
            missingSegments.add(segment.getDescriptor());
        }
        return new LeafSegmentsBundle(segmentReferences, loadableSegments, missingSegments);
    }

    public Optional<Segment> acquireCachedSegment(DataSegment dataSegment) {
        return this.cacheManager.acquireCachedSegment(dataSegment);
    }

    public AcquireSegmentAction acquireSegment(DataSegment dataSegment) throws SegmentLoadingException {
        return this.cacheManager.acquireSegment(dataSegment);
    }

    public Optional<Stream<ReferenceCountedIndexedTableProvider>> getIndexedTables(TableDataSource dataSource) {
        return this.getTimeline(dataSource).map(timeline -> {
            Stream<ReferenceCountedIndexedTableProvider> segments = timeline.lookup(Intervals.ETERNITY).stream().flatMap(x -> StreamSupport.stream(x.getObject().payloads().spliterator(), false));
            ConcurrentHashMap tables = Optional.ofNullable(this.dataSources.get(dataSource.getName())).map(DataSourceState::getTablesLookup).orElseThrow(() -> new ISE("dataSource[%s] does not have IndexedTables", new Object[]{dataSource.getName()}));
            return segments.map(segment -> (ReferenceCountedIndexedTableProvider)tables.get(segment.getId())).filter(Objects::nonNull);
        });
    }

    public boolean hasIndexedTables(String dataSourceName) {
        if (this.dataSources.containsKey(dataSourceName)) {
            return !this.dataSources.get((Object)dataSourceName).tablesLookup.isEmpty();
        }
        return false;
    }

    public void loadSegmentOnBootstrap(DataSegment dataSegment, SegmentLazyLoadFailCallback loadFailed) throws SegmentLoadingException, IOException {
        try {
            this.cacheManager.bootstrap(dataSegment, loadFailed);
        }
        catch (SegmentLoadingException e) {
            this.cacheManager.drop(dataSegment);
            throw e;
        }
        this.loadSegmentInternal(dataSegment);
    }

    public void loadSegment(DataSegment dataSegment) throws SegmentLoadingException, IOException {
        try {
            this.cacheManager.load(dataSegment);
        }
        catch (SegmentLoadingException e) {
            this.cacheManager.drop(dataSegment);
            throw e;
        }
        this.loadSegmentInternal(dataSegment);
    }

    private void loadSegmentInternal(DataSegment dataSegment) throws IOException {
        SettableSupplier resultSupplier = new SettableSupplier();
        this.dataSources.compute(dataSegment.getDataSource(), (k, v) -> {
            DataSourceState dataSourceState = v == null ? new DataSourceState() : v;
            VersionedIntervalTimeline<String, DataSegment> loadedIntervals = dataSourceState.getTimeline();
            PartitionChunk entry = loadedIntervals.findChunk(dataSegment.getInterval(), (Object)dataSegment.getVersion(), dataSegment.getShardSpec().getPartitionNum());
            if (entry != null) {
                log.warn("Told to load segment[%s] that already exists", new Object[]{dataSegment.getId()});
                resultSupplier.set((Object)false);
            } else {
                loadedIntervals.add(dataSegment.getInterval(), (Object)dataSegment.getVersion(), dataSegment.getShardSpec().createChunk((Object)dataSegment));
                long numOfRows = 0L;
                Optional<Segment> loadedSegment = this.cacheManager.acquireCachedSegment(dataSegment);
                if (loadedSegment.isPresent()) {
                    Segment segment = loadedSegment.get();
                    IndexedTable table = (IndexedTable)segment.as(IndexedTable.class);
                    if (table != null) {
                        if (dataSourceState.isEmpty() || dataSourceState.numSegments == (long)dataSourceState.tablesLookup.size()) {
                            dataSourceState.tablesLookup.put(segment.getId(), new ReferenceCountedIndexedTableProvider(table));
                        } else {
                            log.error("Cannot load segment[%s] with IndexedTable, no existing segments are joinable", new Object[]{segment.getId()});
                        }
                    } else if (!dataSourceState.tablesLookup.isEmpty()) {
                        log.error("Cannot load segment[%s] without IndexedTable, all existing segments are joinable", new Object[]{segment.getId()});
                    }
                    PhysicalSegmentInspector countInspector = (PhysicalSegmentInspector)segment.as(PhysicalSegmentInspector.class);
                    if (countInspector != null) {
                        numOfRows = countInspector.getNumRows();
                    }
                    CloseableUtils.closeAndWrapExceptions((Closeable)segment);
                }
                dataSourceState.addSegment(dataSegment, numOfRows);
                resultSupplier.set((Object)true);
            }
            return dataSourceState;
        });
        boolean loadResult = (Boolean)resultSupplier.get();
        if (loadResult) {
            this.cacheManager.storeInfoFile(dataSegment);
        }
    }

    public void dropSegment(DataSegment dataSegment) {
        String dataSource = dataSegment.getDataSource();
        this.dataSources.compute(dataSource, (dataSourceName, dataSourceState) -> {
            block12: {
                DataSegment oldSegmentRef;
                if (dataSourceState == null) {
                    log.info("Told to delete a queryable for a dataSource[%s] that doesn't exist.", new Object[]{dataSourceName});
                    return null;
                }
                VersionedIntervalTimeline<String, DataSegment> loadedIntervals = dataSourceState.getTimeline();
                PartitionChunk removed = loadedIntervals.remove(dataSegment.getInterval(), (Object)dataSegment.getVersion(), dataSegment.getShardSpec().createChunk((Object)dataSegment));
                DataSegment dataSegment2 = oldSegmentRef = removed == null ? null : (DataSegment)removed.getObject();
                if (oldSegmentRef != null) {
                    try (Closer closer = Closer.create();){
                        Optional<Segment> oldSegment = this.cacheManager.acquireCachedSegment(oldSegmentRef);
                        long numberOfRows = oldSegment.map(segment -> {
                            closer.register((Closeable)segment);
                            PhysicalSegmentInspector countInspector = (PhysicalSegmentInspector)segment.as(PhysicalSegmentInspector.class);
                            if (countInspector != null) {
                                return countInspector.getNumRows();
                            }
                            return 0;
                        }).orElse(0).intValue();
                        dataSourceState.removeSegment(dataSegment, numberOfRows);
                        log.info("Attempting to close segment[%s]", new Object[]{dataSegment.getId()});
                        ReferenceCountedIndexedTableProvider oldTable = dataSourceState.tablesLookup.remove(dataSegment.getId());
                        if (oldTable != null) {
                            closer.register((Closeable)oldTable);
                        }
                        break block12;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                log.info("Told to delete a queryable on dataSource[%s] for interval[%s] and version[%s] that I don't have.", new Object[]{dataSourceName, dataSegment.getInterval(), dataSegment.getVersion()});
            }
            if (dataSourceState.isEmpty()) {
                return null;
            }
            DataSourceState dataSourceState2 = dataSourceState;
            return dataSourceState2;
        });
        this.cacheManager.removeInfoFile(dataSegment);
        this.cacheManager.drop(dataSegment);
    }

    public boolean canHandleSegments() {
        return this.cacheManager.canHandleSegments();
    }

    public boolean canLoadSegmentsOnDemand() {
        return this.cacheManager.canLoadSegmentsOnDemand();
    }

    public boolean canLoadSegmentOnDemand(DataSegment dataSegment) {
        return this.cacheManager.canLoadSegmentOnDemand(dataSegment);
    }

    public List<DataSegment> getCachedSegments() throws IOException {
        return this.cacheManager.getCachedSegments();
    }

    public void shutdownBootstrap() {
        this.cacheManager.shutdownBootstrap();
    }

    public void shutdown() {
        this.cacheManager.shutdown();
    }

    public static class DataSourceState {
        private final VersionedIntervalTimeline<String, DataSegment> timeline = new VersionedIntervalTimeline((Comparator)Ordering.natural());
        private final ConcurrentHashMap<SegmentId, ReferenceCountedIndexedTableProvider> tablesLookup = new ConcurrentHashMap();
        private long totalSegmentSize;
        private long numSegments;
        private long rowCount;
        private final SegmentRowCountDistribution segmentRowCountDistribution = new SegmentRowCountDistribution();

        private void addSegment(DataSegment segment, long numOfRows) {
            this.totalSegmentSize += segment.getSize();
            ++this.numSegments;
            this.rowCount += numOfRows;
            if (segment.isTombstone()) {
                this.segmentRowCountDistribution.addTombstoneToDistribution();
            } else {
                this.segmentRowCountDistribution.addRowCountToDistribution(numOfRows);
            }
        }

        private void removeSegment(DataSegment segment, long numOfRows) {
            this.totalSegmentSize -= segment.getSize();
            --this.numSegments;
            this.rowCount -= numOfRows;
            if (segment.isTombstone()) {
                this.segmentRowCountDistribution.removeTombstoneFromDistribution();
            } else {
                this.segmentRowCountDistribution.removeRowCountFromDistribution(numOfRows);
            }
        }

        public VersionedIntervalTimeline<String, DataSegment> getTimeline() {
            return this.timeline;
        }

        public ConcurrentHashMap<SegmentId, ReferenceCountedIndexedTableProvider> getTablesLookup() {
            return this.tablesLookup;
        }

        public long getAverageRowCount() {
            return this.numSegments == 0L ? 0L : this.rowCount / this.numSegments;
        }

        public long getTotalSegmentSize() {
            return this.totalSegmentSize;
        }

        public long getNumSegments() {
            return this.numSegments;
        }

        public boolean isEmpty() {
            return this.numSegments == 0L;
        }

        private SegmentRowCountDistribution getSegmentRowCountDistribution() {
            return this.segmentRowCountDistribution;
        }
    }
}

