/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.telemetry.internals;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Predicate;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.message.GetTelemetrySubscriptionsRequestData;
import org.apache.kafka.common.message.GetTelemetrySubscriptionsResponseData;
import org.apache.kafka.common.message.PushTelemetryRequestData;
import org.apache.kafka.common.message.PushTelemetryResponseData;
import org.apache.kafka.common.metrics.KafkaMetric;
import org.apache.kafka.common.metrics.MetricsContext;
import org.apache.kafka.common.metrics.MetricsReporter;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.GetTelemetrySubscriptionsRequest;
import org.apache.kafka.common.requests.GetTelemetrySubscriptionsResponse;
import org.apache.kafka.common.requests.PushTelemetryRequest;
import org.apache.kafka.common.requests.PushTelemetryResponse;
import org.apache.kafka.common.telemetry.ClientTelemetryState;
import org.apache.kafka.common.telemetry.internals.ClientTelemetryEmitter;
import org.apache.kafka.common.telemetry.internals.ClientTelemetryProvider;
import org.apache.kafka.common.telemetry.internals.ClientTelemetrySender;
import org.apache.kafka.common.telemetry.internals.ClientTelemetryUtils;
import org.apache.kafka.common.telemetry.internals.KafkaMetricsCollector;
import org.apache.kafka.common.telemetry.internals.MetricKeyable;
import org.apache.kafka.common.telemetry.internals.MetricsCollector;
import org.apache.kafka.common.telemetry.internals.SinglePointMetric;
import org.apache.kafka.common.telemetry.internals.TelemetryMetricNamingConvention;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.shaded.io.opentelemetry.proto.metrics.v1.Metric;
import org.apache.kafka.shaded.io.opentelemetry.proto.metrics.v1.MetricsData;
import org.apache.kafka.shaded.io.opentelemetry.proto.metrics.v1.ResourceMetrics;
import org.apache.kafka.shaded.io.opentelemetry.proto.metrics.v1.ScopeMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientTelemetryReporter
implements MetricsReporter {
    private static final Logger log = LoggerFactory.getLogger(ClientTelemetryReporter.class);
    private static final Set<String> EXCLUDE_LABELS = Collections.singleton("client_id");
    public static final int DEFAULT_PUSH_INTERVAL_MS = 300000;
    private final ClientTelemetryProvider telemetryProvider;
    private final ClientTelemetrySender clientTelemetrySender;
    private final Time time;
    private Map<String, Object> rawOriginalConfig;
    private KafkaMetricsCollector kafkaMetricsCollector;

    public ClientTelemetryReporter(Time time) {
        this.time = time;
        this.telemetryProvider = new ClientTelemetryProvider();
        this.clientTelemetrySender = new DefaultClientTelemetrySender();
    }

    @Override
    public synchronized void configure(Map<String, ?> configs) {
        this.rawOriginalConfig = Objects.requireNonNull(configs);
    }

    @Override
    public synchronized void contextChange(MetricsContext metricsContext) {
        Objects.requireNonNull(this.rawOriginalConfig, "configure() was not called before contextChange()");
        if (this.kafkaMetricsCollector != null) {
            this.kafkaMetricsCollector.stop();
        }
        if (!this.telemetryProvider.validate(metricsContext)) {
            log.warn("Validation failed for {} context {}, skip starting collectors. Metrics collection is disabled", this.telemetryProvider.getClass(), metricsContext.contextLabels());
            return;
        }
        if (this.kafkaMetricsCollector == null) {
            this.telemetryProvider.configure(this.rawOriginalConfig);
        }
        this.telemetryProvider.contextChange(metricsContext);
        if (this.kafkaMetricsCollector == null) {
            this.initCollectors();
        }
    }

    @Override
    public void init(List<KafkaMetric> metrics) {
        if (this.kafkaMetricsCollector != null) {
            this.kafkaMetricsCollector.init(metrics);
        }
    }

    @Override
    public void metricChange(KafkaMetric metric) {
        if (this.kafkaMetricsCollector != null) {
            this.kafkaMetricsCollector.metricChange(metric);
        }
    }

    @Override
    public void metricRemoval(KafkaMetric metric) {
        if (this.kafkaMetricsCollector != null) {
            this.kafkaMetricsCollector.metricRemoval(metric);
        }
    }

    @Override
    public void close() {
        log.debug("Stopping ClientTelemetryReporter");
        try {
            this.clientTelemetrySender.close();
        }
        catch (Exception exception) {
            log.error("Failed to close client telemetry reporter", (Throwable)exception);
        }
    }

    public synchronized void updateMetricsLabels(Map<String, String> labels) {
        this.telemetryProvider.updateLabels(labels);
    }

    public void initiateClose() {
        log.debug("Initiate close of ClientTelemetryReporter");
        try {
            this.clientTelemetrySender.initiateClose();
        }
        catch (Exception exception) {
            log.error("Failed to initiate close of client telemetry reporter", (Throwable)exception);
        }
    }

    public ClientTelemetrySender telemetrySender() {
        return this.clientTelemetrySender;
    }

    private void initCollectors() {
        this.kafkaMetricsCollector = new KafkaMetricsCollector(TelemetryMetricNamingConvention.getClientTelemetryMetricNamingStrategy(this.telemetryProvider.domain()), EXCLUDE_LABELS);
    }

    private ResourceMetrics buildMetric(Metric metric) {
        return ResourceMetrics.newBuilder().setResource(this.telemetryProvider.resource()).addScopeMetrics(ScopeMetrics.newBuilder().addMetrics(metric).build()).build();
    }

    void metricsCollector(KafkaMetricsCollector metricsCollector) {
        this.kafkaMetricsCollector = metricsCollector;
    }

    MetricsCollector metricsCollector() {
        return this.kafkaMetricsCollector;
    }

    ClientTelemetryProvider telemetryProvider() {
        return this.telemetryProvider;
    }

    class DefaultClientTelemetrySender
    implements ClientTelemetrySender {
        private static final double INITIAL_PUSH_JITTER_LOWER = 0.5;
        private static final double INITIAL_PUSH_JITTER_UPPER = 1.5;
        private final Set<CompressionType> unsupportedCompressionTypes = ConcurrentHashMap.newKeySet();
        private final ReadWriteLock lock = new ReentrantReadWriteLock();
        private final Condition subscriptionLoaded = this.lock.writeLock().newCondition();
        private ClientTelemetryState state = ClientTelemetryState.SUBSCRIPTION_NEEDED;
        private ClientTelemetrySubscription subscription;
        private long lastRequestMs;
        private int intervalMs;
        private boolean enabled = true;

        private DefaultClientTelemetrySender() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public long timeToNextUpdate(long requestTimeoutMs) {
            long timeMs;
            int localIntervalMs;
            long localLastRequestMs;
            ClientTelemetryState localState;
            long nowMs = ClientTelemetryReporter.this.time.milliseconds();
            this.lock.readLock().lock();
            try {
                if (!this.enabled) {
                    long l = Integer.MAX_VALUE;
                    return l;
                }
                localState = this.state;
                localLastRequestMs = this.lastRequestMs;
                localIntervalMs = this.intervalMs;
            }
            finally {
                this.lock.readLock().unlock();
            }
            switch (localState) {
                case SUBSCRIPTION_IN_PROGRESS: 
                case PUSH_IN_PROGRESS: {
                    String apiName = localState == ClientTelemetryState.SUBSCRIPTION_IN_PROGRESS ? ApiKeys.GET_TELEMETRY_SUBSCRIPTIONS.name : ApiKeys.PUSH_TELEMETRY.name;
                    timeMs = requestTimeoutMs;
                    log.trace("For telemetry state {}, returning the value {} ms; the remaining wait time for the {} network API request, as specified by {}", new Object[]{localState, timeMs, apiName, "request.timeout.ms"});
                    break;
                }
                case TERMINATING_PUSH_IN_PROGRESS: {
                    timeMs = Long.MAX_VALUE;
                    log.trace("For telemetry state {}, returning the value {} ms; the terminating push is in progress, disabling telemetry for further requests", (Object)localState, (Object)timeMs);
                    break;
                }
                case TERMINATED: {
                    timeMs = Long.MAX_VALUE;
                    log.trace("For telemetry state {}, returning the value {} ms; telemetry is terminated, no further requests will be made", (Object)localState, (Object)timeMs);
                    break;
                }
                case TERMINATING_PUSH_NEEDED: {
                    timeMs = 0L;
                    log.trace("For telemetry state {}, returning the value {} ms; the client should try to submit the final {} network API request ASAP before closing", new Object[]{localState, timeMs, ApiKeys.PUSH_TELEMETRY.name});
                    break;
                }
                case SUBSCRIPTION_NEEDED: 
                case PUSH_NEEDED: {
                    String apiName = localState == ClientTelemetryState.SUBSCRIPTION_NEEDED ? ApiKeys.GET_TELEMETRY_SUBSCRIPTIONS.name : ApiKeys.PUSH_TELEMETRY.name;
                    long timeRemainingBeforeRequest = localLastRequestMs + (long)localIntervalMs - nowMs;
                    if (timeRemainingBeforeRequest <= 0L) {
                        timeMs = 0L;
                        log.trace("For telemetry state {}, returning the value {} ms; the wait time before submitting the next {} network API request has elapsed", new Object[]{localState, timeMs, apiName});
                        break;
                    }
                    timeMs = timeRemainingBeforeRequest;
                    log.trace("For telemetry state {}, returning the value {} ms; the client will wait before submitting the next {} network API request", new Object[]{localState, timeMs, apiName});
                    break;
                }
                default: {
                    throw new IllegalStateException("Unknown telemetry state: " + String.valueOf((Object)localState));
                }
            }
            return timeMs;
        }

        @Override
        public Optional<AbstractRequest.Builder<?>> createRequest() {
            ClientTelemetrySubscription localSubscription;
            ClientTelemetryState localState;
            this.lock.readLock().lock();
            try {
                localState = this.state;
                localSubscription = this.subscription;
            }
            finally {
                this.lock.readLock().unlock();
            }
            if (localState == ClientTelemetryState.SUBSCRIPTION_NEEDED) {
                return this.createSubscriptionRequest(localSubscription);
            }
            if (localState == ClientTelemetryState.PUSH_NEEDED || localState == ClientTelemetryState.TERMINATING_PUSH_NEEDED) {
                return this.createPushRequest(localSubscription);
            }
            log.warn("Cannot make telemetry request as telemetry is in state: {}", (Object)localState);
            return Optional.empty();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handleResponse(GetTelemetrySubscriptionsResponse response) {
            ClientTelemetrySubscription oldSubscription;
            ClientTelemetryState oldState;
            long nowMs = ClientTelemetryReporter.this.time.milliseconds();
            GetTelemetrySubscriptionsResponseData data = response.data();
            this.lock.readLock().lock();
            try {
                oldState = this.state;
                oldSubscription = this.subscription;
            }
            finally {
                this.lock.readLock().unlock();
            }
            Optional<Integer> errorIntervalMsOpt = ClientTelemetryUtils.maybeFetchErrorIntervalMs(data.errorCode(), oldSubscription != null ? oldSubscription.pushIntervalMs() : -1);
            if (errorIntervalMsOpt.isPresent()) {
                if (!this.maybeSetState(ClientTelemetryState.SUBSCRIPTION_NEEDED)) {
                    log.warn("Unable to transition state after failed get telemetry subscriptions from state {}", (Object)oldState);
                }
                this.updateErrorResult(errorIntervalMsOpt.get(), nowMs);
                return;
            }
            Uuid clientInstanceId = ClientTelemetryUtils.validateClientInstanceId(data.clientInstanceId());
            int intervalMs = ClientTelemetryUtils.validateIntervalMs(data.pushIntervalMs());
            Predicate<? super MetricKeyable> selector = ClientTelemetryUtils.getSelectorFromRequestedMetrics(data.requestedMetrics());
            List<CompressionType> acceptedCompressionTypes = ClientTelemetryUtils.getCompressionTypesFromAcceptedList(data.acceptedCompressionTypes());
            if (oldSubscription != null && oldSubscription.deltaTemporality() != data.deltaTemporality()) {
                log.info("Delta temporality has changed from {} to {}, resetting metric values", (Object)oldSubscription.deltaTemporality(), (Object)data.deltaTemporality());
                if (ClientTelemetryReporter.this.kafkaMetricsCollector != null) {
                    ClientTelemetryReporter.this.kafkaMetricsCollector.metricsReset();
                }
            }
            ClientTelemetrySubscription clientTelemetrySubscription = new ClientTelemetrySubscription(clientInstanceId, data.subscriptionId(), intervalMs, acceptedCompressionTypes, data.deltaTemporality(), selector);
            this.lock.writeLock().lock();
            try {
                if (this.isTerminatingState()) {
                    return;
                }
                ClientTelemetryState newState = selector == ClientTelemetryUtils.SELECTOR_NO_METRICS ? ClientTelemetryState.SUBSCRIPTION_NEEDED : ClientTelemetryState.PUSH_NEEDED;
                if (!this.maybeSetState(newState)) {
                    return;
                }
                this.updateSubscriptionResult(clientTelemetrySubscription, nowMs);
                log.info("Client telemetry registered with client instance id: {}", (Object)this.subscription.clientInstanceId());
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handleResponse(PushTelemetryResponse response) {
            long nowMs = ClientTelemetryReporter.this.time.milliseconds();
            PushTelemetryResponseData data = response.data();
            this.lock.writeLock().lock();
            try {
                if (this.isTerminatingState()) {
                    return;
                }
                Optional<Integer> errorIntervalMsOpt = ClientTelemetryUtils.maybeFetchErrorIntervalMs(data.errorCode(), this.subscription.pushIntervalMs());
                if (errorIntervalMsOpt.isPresent()) {
                    if (!this.maybeSetState(ClientTelemetryState.SUBSCRIPTION_NEEDED)) {
                        log.warn("Unable to transition state after failed push telemetry from state {}", (Object)this.state);
                    }
                    this.updateErrorResult(errorIntervalMsOpt.get(), nowMs);
                    return;
                }
                this.lastRequestMs = nowMs;
                this.intervalMs = this.subscription.pushIntervalMs();
                if (!this.maybeSetState(ClientTelemetryState.PUSH_NEEDED)) {
                    log.warn("Unable to transition state after successful push telemetry from state {}", (Object)this.state);
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        @Override
        public void handleFailedGetTelemetrySubscriptionsRequest(KafkaException maybeFatalException) {
            log.debug("The broker generated an error for the get telemetry network API request", (Throwable)maybeFatalException);
            this.handleFailedRequest(maybeFatalException);
        }

        @Override
        public void handleFailedPushTelemetryRequest(KafkaException maybeFatalException) {
            log.debug("The broker generated an error for the push telemetry network API request", (Throwable)maybeFatalException);
            this.handleFailedRequest(maybeFatalException);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Optional<Uuid> clientInstanceId(Duration timeout) {
            long timeoutMs = timeout.toMillis();
            if (timeoutMs < 0L) {
                throw new IllegalArgumentException("The timeout cannot be negative for fetching client instance id.");
            }
            this.lock.writeLock().lock();
            try {
                if (this.subscription == null) {
                    log.debug("Waiting for telemetry subscription containing the client instance ID with timeoutMillis = {} ms.", (Object)timeoutMs);
                    try {
                        if (!this.subscriptionLoaded.await(timeoutMs, TimeUnit.MILLISECONDS)) {
                            log.debug("Wait for telemetry subscription elapsed; may not have actually loaded it");
                        }
                    }
                    catch (InterruptedException e) {
                        throw new InterruptException(e);
                    }
                }
                if (this.subscription == null) {
                    log.debug("Client instance ID could not be retrieved with timeout {}", (Object)timeout);
                    Optional<Uuid> e = Optional.empty();
                    return e;
                }
                Uuid clientInstanceId = this.subscription.clientInstanceId();
                if (clientInstanceId == null) {
                    log.info("Client instance ID was null in telemetry subscription while in state {}", (Object)this.state);
                    Optional<Uuid> optional = Optional.empty();
                    return optional;
                }
                Optional<Uuid> optional = Optional.of(clientInstanceId);
                return optional;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        @Override
        public void close() {
            log.debug("close telemetry sender for client telemetry reporter instance");
            boolean shouldClose = false;
            this.lock.writeLock().lock();
            try {
                if (this.state != ClientTelemetryState.TERMINATED) {
                    if (this.maybeSetState(ClientTelemetryState.TERMINATED)) {
                        shouldClose = true;
                    }
                } else {
                    log.debug("Ignoring subsequent close");
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
            if (shouldClose && ClientTelemetryReporter.this.kafkaMetricsCollector != null) {
                ClientTelemetryReporter.this.kafkaMetricsCollector.stop();
            }
        }

        @Override
        public void initiateClose() {
            log.debug("initiate close for client telemetry, check if terminal push required.");
            this.lock.writeLock().lock();
            try {
                if (this.lastRequestMs == 0L) {
                    log.debug("Telemetry subscription not loaded, not attempting terminating push");
                    return;
                }
                if (this.state == ClientTelemetryState.SUBSCRIPTION_NEEDED) {
                    log.debug("Subscription not yet loaded, ignoring terminal push");
                    return;
                }
                if (this.isTerminatingState() || !this.maybeSetState(ClientTelemetryState.TERMINATING_PUSH_NEEDED)) {
                    log.debug("Ignoring subsequent initiateClose");
                    return;
                }
                log.debug("Updated state to send terminal telemetry push request");
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        private boolean isRetryable(KafkaException maybeFatalException) {
            return maybeFatalException == null || maybeFatalException instanceof RetriableException || maybeFatalException.getCause() != null && maybeFatalException.getCause() instanceof RetriableException;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Optional<AbstractRequest.Builder<?>> createSubscriptionRequest(ClientTelemetrySubscription localSubscription) {
            Uuid clientInstanceId = localSubscription != null ? localSubscription.clientInstanceId() : Uuid.ZERO_UUID;
            log.debug("Creating telemetry subscription request with client instance id {}", (Object)clientInstanceId);
            this.lock.writeLock().lock();
            try {
                if (this.isTerminatingState()) {
                    Optional<AbstractRequest.Builder<?>> optional = Optional.empty();
                    return optional;
                }
                if (!this.maybeSetState(ClientTelemetryState.SUBSCRIPTION_IN_PROGRESS)) {
                    Optional<AbstractRequest.Builder<?>> optional = Optional.empty();
                    return optional;
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
            GetTelemetrySubscriptionsRequest.Builder requestBuilder = new GetTelemetrySubscriptionsRequest.Builder(new GetTelemetrySubscriptionsRequestData().setClientInstanceId(clientInstanceId), true);
            return Optional.of(requestBuilder);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Optional<AbstractRequest.Builder<?>> createPushRequest(ClientTelemetrySubscription localSubscription) {
            boolean terminating;
            if (localSubscription == null) {
                log.warn("Telemetry state is {} but subscription is null; not sending telemetry", (Object)this.state);
                if (!this.maybeSetState(ClientTelemetryState.SUBSCRIPTION_NEEDED)) {
                    log.warn("Unable to transition state after failed create push telemetry from state {}", (Object)this.state);
                }
                return Optional.empty();
            }
            log.debug("Creating telemetry push request with client instance id {}", (Object)localSubscription.clientInstanceId());
            if (ClientTelemetryReporter.this.kafkaMetricsCollector == null) {
                log.warn("Cannot make telemetry request as collector is not initialized");
                this.updateErrorResult(localSubscription.pushIntervalMs, ClientTelemetryReporter.this.time.milliseconds());
                return Optional.empty();
            }
            this.lock.writeLock().lock();
            try {
                if (this.state == ClientTelemetryState.TERMINATED || this.state == ClientTelemetryState.TERMINATING_PUSH_IN_PROGRESS) {
                    Optional<AbstractRequest.Builder<?>> optional = Optional.empty();
                    return optional;
                }
                terminating = this.state == ClientTelemetryState.TERMINATING_PUSH_NEEDED;
                if (!this.maybeSetState(terminating ? ClientTelemetryState.TERMINATING_PUSH_IN_PROGRESS : ClientTelemetryState.PUSH_IN_PROGRESS)) {
                    Optional<AbstractRequest.Builder<?>> optional = Optional.empty();
                    return optional;
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
            return this.createPushRequest(localSubscription, terminating);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private Optional<AbstractRequest.Builder<?>> createPushRequest(ClientTelemetrySubscription localSubscription, boolean terminating) {
            ByteBuffer compressedPayload;
            MetricsData payload;
            try (ClientTelemetryEmitter emitter = new ClientTelemetryEmitter(localSubscription.selector(), localSubscription.deltaTemporality());){
                emitter.init();
                ClientTelemetryReporter.this.kafkaMetricsCollector.collect(emitter);
                payload = this.createPayload(emitter.emittedMetrics());
            }
            catch (Exception e) {
                log.warn("Error constructing client telemetry payload: ", (Throwable)e);
                this.updateErrorResult(localSubscription.pushIntervalMs, ClientTelemetryReporter.this.time.milliseconds());
                return Optional.empty();
            }
            CompressionType compressionType = ClientTelemetryUtils.preferredCompressionType(localSubscription.acceptedCompressionTypes(), this.unsupportedCompressionTypes);
            try {
                compressedPayload = ClientTelemetryUtils.compress(payload, compressionType);
            }
            catch (Throwable e) {
                if (e instanceof Error && !(e instanceof NoClassDefFoundError) && !(e.getCause() instanceof NoClassDefFoundError)) {
                    this.lock.writeLock().lock();
                    try {
                        this.state = ClientTelemetryState.TERMINATED;
                    }
                    finally {
                        this.lock.writeLock().unlock();
                    }
                    log.error("Unexpected error occurred while compressing telemetry payload for compression: {}, stopping client telemetry", (Object)compressionType, (Object)e);
                    throw new KafkaException("Unexpected compression error", e);
                }
                log.debug("Failed to compress telemetry payload for compression: {}, sending uncompressed data", (Object)compressionType, (Object)e);
                this.unsupportedCompressionTypes.add(compressionType);
                compressedPayload = ByteBuffer.wrap(payload.toByteArray());
                compressionType = CompressionType.NONE;
            }
            PushTelemetryRequest.Builder requestBuilder = new PushTelemetryRequest.Builder(new PushTelemetryRequestData().setClientInstanceId(localSubscription.clientInstanceId()).setSubscriptionId(localSubscription.subscriptionId()).setTerminating(terminating).setCompressionType(compressionType.id).setMetrics(compressedPayload), true);
            return Optional.of(requestBuilder);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void updateSubscriptionResult(ClientTelemetrySubscription subscription, long timeMs) {
            this.lock.writeLock().lock();
            try {
                this.subscription = Objects.requireNonNull(subscription);
                this.intervalMs = this.state == ClientTelemetryState.PUSH_NEEDED ? this.computeStaggeredIntervalMs(subscription.pushIntervalMs(), 0.5, 1.5) : subscription.pushIntervalMs();
                this.lastRequestMs = timeMs;
                log.debug("Updating subscription - subscription: {}; intervalMs: {}, lastRequestMs: {}", new Object[]{subscription, this.intervalMs, this.lastRequestMs});
                this.subscriptionLoaded.signalAll();
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void updateErrorResult(int intervalMs, long timeMs) {
            this.lock.writeLock().lock();
            try {
                this.intervalMs = intervalMs;
                this.lastRequestMs = timeMs;
                if (intervalMs == Integer.MAX_VALUE) {
                    this.enabled = false;
                }
                log.debug("Updating intervalMs: {}, lastRequestMs: {}", (Object)intervalMs, (Object)this.lastRequestMs);
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        int computeStaggeredIntervalMs(int intervalMs, double lowerBound, double upperBound) {
            double rand = ThreadLocalRandom.current().nextDouble(lowerBound, upperBound);
            int firstPushIntervalMs = (int)Math.round(rand * (double)intervalMs);
            log.debug("Telemetry subscription push interval value from broker was {}; to stagger requests the first push interval is being adjusted to {}", (Object)intervalMs, (Object)firstPushIntervalMs);
            return firstPushIntervalMs;
        }

        private boolean isTerminatingState() {
            return this.state == ClientTelemetryState.TERMINATED || this.state == ClientTelemetryState.TERMINATING_PUSH_NEEDED || this.state == ClientTelemetryState.TERMINATING_PUSH_IN_PROGRESS;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        boolean maybeSetState(ClientTelemetryState newState) {
            this.lock.writeLock().lock();
            try {
                ClientTelemetryState oldState = this.state;
                this.state = oldState.validateTransition(newState);
                log.debug("Setting telemetry state from {} to {}", (Object)oldState, (Object)newState);
                boolean bl = true;
                return bl;
            }
            catch (IllegalStateException e) {
                log.warn("Error updating client telemetry state, disabled telemetry");
                this.enabled = false;
                boolean bl = false;
                return bl;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void handleFailedRequest(KafkaException maybeFatalException) {
            long nowMs = ClientTelemetryReporter.this.time.milliseconds();
            this.lock.writeLock().lock();
            try {
                if (this.isTerminatingState()) {
                    return;
                }
                if (this.state != ClientTelemetryState.SUBSCRIPTION_IN_PROGRESS && this.state != ClientTelemetryState.PUSH_IN_PROGRESS) {
                    log.warn("Could not transition state after failed telemetry from state {}, disabling telemetry", (Object)this.state);
                    this.updateErrorResult(Integer.MAX_VALUE, nowMs);
                    return;
                }
                if (this.isRetryable(maybeFatalException)) {
                    this.updateErrorResult(300000, nowMs);
                } else {
                    if (!(maybeFatalException instanceof UnsupportedVersionException)) {
                        log.warn("Received unrecoverable error from broker, disabling telemetry");
                    }
                    this.updateErrorResult(Integer.MAX_VALUE, nowMs);
                }
                if (!this.maybeSetState(ClientTelemetryState.SUBSCRIPTION_NEEDED)) {
                    log.warn("Could not transition state after failed telemetry from state {}", (Object)this.state);
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        private MetricsData createPayload(List<SinglePointMetric> emittedMetrics) {
            MetricsData.Builder builder = MetricsData.newBuilder();
            emittedMetrics.forEach(metric -> {
                Metric m = metric.builder().build();
                ResourceMetrics rm = ClientTelemetryReporter.this.buildMetric(m);
                builder.addResourceMetrics(rm);
            });
            return builder.build();
        }

        ClientTelemetrySubscription subscription() {
            this.lock.readLock().lock();
            try {
                ClientTelemetrySubscription clientTelemetrySubscription = this.subscription;
                return clientTelemetrySubscription;
            }
            finally {
                this.lock.readLock().unlock();
            }
        }

        ClientTelemetryState state() {
            this.lock.readLock().lock();
            try {
                ClientTelemetryState clientTelemetryState = this.state;
                return clientTelemetryState;
            }
            finally {
                this.lock.readLock().unlock();
            }
        }

        long intervalMs() {
            this.lock.readLock().lock();
            try {
                long l = this.intervalMs;
                return l;
            }
            finally {
                this.lock.readLock().unlock();
            }
        }

        long lastRequestMs() {
            this.lock.readLock().lock();
            try {
                long l = this.lastRequestMs;
                return l;
            }
            finally {
                this.lock.readLock().unlock();
            }
        }

        void enabled(boolean enabled) {
            this.lock.writeLock().lock();
            try {
                this.enabled = enabled;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }

        boolean enabled() {
            this.lock.readLock().lock();
            try {
                boolean bl = this.enabled;
                return bl;
            }
            finally {
                this.lock.readLock().unlock();
            }
        }
    }

    static class ClientTelemetrySubscription {
        private final Uuid clientInstanceId;
        private final int subscriptionId;
        private final int pushIntervalMs;
        private final List<CompressionType> acceptedCompressionTypes;
        private final boolean deltaTemporality;
        private final Predicate<? super MetricKeyable> selector;

        ClientTelemetrySubscription(Uuid clientInstanceId, int subscriptionId, int pushIntervalMs, List<CompressionType> acceptedCompressionTypes, boolean deltaTemporality, Predicate<? super MetricKeyable> selector) {
            this.clientInstanceId = clientInstanceId;
            this.subscriptionId = subscriptionId;
            this.pushIntervalMs = pushIntervalMs;
            this.acceptedCompressionTypes = Collections.unmodifiableList(acceptedCompressionTypes);
            this.deltaTemporality = deltaTemporality;
            this.selector = selector;
        }

        public Uuid clientInstanceId() {
            return this.clientInstanceId;
        }

        public int subscriptionId() {
            return this.subscriptionId;
        }

        public int pushIntervalMs() {
            return this.pushIntervalMs;
        }

        public List<CompressionType> acceptedCompressionTypes() {
            return this.acceptedCompressionTypes;
        }

        public boolean deltaTemporality() {
            return this.deltaTemporality;
        }

        public Predicate<? super MetricKeyable> selector() {
            return this.selector;
        }

        public String toString() {
            return new StringJoiner(", ", "ClientTelemetrySubscription{", "}").add("clientInstanceId=" + String.valueOf(this.clientInstanceId)).add("subscriptionId=" + this.subscriptionId).add("pushIntervalMs=" + this.pushIntervalMs).add("acceptedCompressionTypes=" + String.valueOf(this.acceptedCompressionTypes)).add("deltaTemporality=" + this.deltaTemporality).add("selector=" + String.valueOf(this.selector)).toString();
        }
    }
}

