/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { ExtensionManager } from '../../config/extension-manager.js';
import { loadSettings } from '../../config/settings.js';
import { requestConsentNonInteractive } from '../../config/extensions/consent.js';
import { debugLogger, } from '@google/gemini-cli-core';
import prompts from 'prompts';
import { promptForSetting, updateSetting, getScopedEnvContents, ExtensionSettingScope, } from '../../config/extensions/extensionSettings.js';
const defaultLogger = {
    log: (message) => debugLogger.log(message),
    error: (message) => debugLogger.error(message),
};
const defaultRequestSetting = async (setting) => promptForSetting(setting);
const defaultRequestConfirmation = async (message) => {
    const response = await prompts({
        type: 'confirm',
        name: 'confirm',
        message,
        initial: false,
    });
    return response.confirm;
};
export async function getExtensionManager() {
    const workspaceDir = process.cwd();
    const extensionManager = new ExtensionManager({
        workspaceDir,
        requestConsent: requestConsentNonInteractive,
        requestSetting: promptForSetting,
        settings: loadSettings(workspaceDir).merged,
    });
    await extensionManager.loadExtensions();
    return extensionManager;
}
export async function getExtensionAndManager(extensionManager, name, logger = defaultLogger) {
    const extension = extensionManager
        .getExtensions()
        .find((ext) => ext.name === name);
    if (!extension) {
        logger.error(`Extension "${name}" is not installed.`);
        return { extension: null };
    }
    return { extension };
}
export async function configureSpecificSetting(extensionManager, extensionName, settingKey, scope, logger = defaultLogger, requestSetting = defaultRequestSetting) {
    const { extension } = await getExtensionAndManager(extensionManager, extensionName, logger);
    if (!extension) {
        return;
    }
    const extensionConfig = await extensionManager.loadExtensionConfig(extension.path);
    if (!extensionConfig) {
        logger.error(`Could not find configuration for extension "${extensionName}".`);
        return;
    }
    await updateSetting(extensionConfig, extension.id, settingKey, requestSetting, scope, process.cwd());
    logger.log(`Setting "${settingKey}" updated.`);
}
export async function configureExtension(extensionManager, extensionName, scope, logger = defaultLogger, requestSetting = defaultRequestSetting, requestConfirmation = defaultRequestConfirmation) {
    const { extension } = await getExtensionAndManager(extensionManager, extensionName, logger);
    if (!extension) {
        return;
    }
    const extensionConfig = await extensionManager.loadExtensionConfig(extension.path);
    if (!extensionConfig ||
        !extensionConfig.settings ||
        extensionConfig.settings.length === 0) {
        logger.log(`Extension "${extensionName}" has no settings to configure.`);
        return;
    }
    logger.log(`Configuring settings for "${extensionName}"...`);
    await configureExtensionSettings(extensionConfig, extension.id, scope, logger, requestSetting, requestConfirmation);
}
export async function configureAllExtensions(extensionManager, scope, logger = defaultLogger, requestSetting = defaultRequestSetting, requestConfirmation = defaultRequestConfirmation) {
    const extensions = extensionManager.getExtensions();
    if (extensions.length === 0) {
        logger.log('No extensions installed.');
        return;
    }
    for (const extension of extensions) {
        const extensionConfig = await extensionManager.loadExtensionConfig(extension.path);
        if (extensionConfig &&
            extensionConfig.settings &&
            extensionConfig.settings.length > 0) {
            logger.log(`\nConfiguring settings for "${extension.name}"...`);
            await configureExtensionSettings(extensionConfig, extension.id, scope, logger, requestSetting, requestConfirmation);
        }
    }
}
export async function configureExtensionSettings(extensionConfig, extensionId, scope, logger = defaultLogger, requestSetting = defaultRequestSetting, requestConfirmation = defaultRequestConfirmation) {
    const currentScopedSettings = await getScopedEnvContents(extensionConfig, extensionId, scope, process.cwd());
    let workspaceSettings = {};
    if (scope === ExtensionSettingScope.USER) {
        workspaceSettings = await getScopedEnvContents(extensionConfig, extensionId, ExtensionSettingScope.WORKSPACE, process.cwd());
    }
    if (!extensionConfig.settings)
        return;
    for (const setting of extensionConfig.settings) {
        const currentValue = currentScopedSettings[setting.envVar];
        const workspaceValue = workspaceSettings[setting.envVar];
        if (workspaceValue !== undefined) {
            logger.log(`Note: Setting "${setting.name}" is already configured in the workspace scope.`);
        }
        if (currentValue !== undefined) {
            const confirmed = await requestConfirmation(`Setting "${setting.name}" (${setting.envVar}) is already set. Overwrite?`);
            if (!confirmed) {
                continue;
            }
        }
        await updateSetting(extensionConfig, extensionId, setting.envVar, requestSetting, scope, process.cwd());
    }
}
export function getFormattedSettingValue(setting) {
    if (!setting.value) {
        return '[not set]';
    }
    if (setting.sensitive) {
        return '***';
    }
    return setting.value;
}
//# sourceMappingURL=utils.js.map