"use strict";
/*
Copyright 2019 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.IrcClientConfig = void 0;
/**
 * Contains IRC client configuration, mostly set by Matrix users. Used to configure
 * IrcUsers.
 */
class IrcClientConfig {
    userId;
    domain;
    config;
    /**
     * Construct an IRC Client Config.
     * @param {string} userId The user ID who is configuring this config.
     * @param {string} domain The IRC network domain for the IRC client
     * @param {Object} configObj Serialised config information if known.
     */
    constructor(userId, domain, config = {}) {
        this.userId = userId;
        this.domain = domain;
        this.config = config;
    }
    getDomain() {
        return this.domain;
    }
    getUserId() {
        return this.userId;
    }
    setUsername(uname) {
        this.config.username = uname;
    }
    getUsername() {
        return this.config.username;
    }
    setPassword(password) {
        this.config.password = password;
    }
    getPassword() {
        return this.config.password;
    }
    setDesiredNick(nick) {
        this.config.nick = nick;
    }
    getDesiredNick() {
        return this.config.nick;
    }
    setIpv6Address(address) {
        this.config.ipv6 = address;
    }
    getIpv6Address() {
        return this.config.ipv6;
    }
    serialize(removePassword = false) {
        if (removePassword) {
            const clone = JSON.parse(JSON.stringify(this.config));
            delete clone.password;
            return clone;
        }
        return this.config;
    }
    toString() {
        const redactedConfig = {
            username: this.config.username,
            nick: this.config.nick,
            ipv6: this.config.ipv6,
            password: this.config.password ? '<REDACTED>' : undefined,
        };
        return this.userId + "=>" + this.domain + "=" + JSON.stringify(redactedConfig);
    }
    static newConfig(matrixUser, domain, nick, username, password) {
        return new IrcClientConfig(matrixUser ? matrixUser.getId() : null, domain, {
            nick: nick,
            username: username,
            password: password
        });
    }
}
exports.IrcClientConfig = IrcClientConfig;
//# sourceMappingURL=IrcClientConfig.js.map