﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/Personalize_EXPORTS.h>
#include <aws/personalize/model/CampaignConfig.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Personalize {
namespace Model {

/**
 * <p>Provides a summary of the properties of a campaign update. For a complete
 * listing, call the <a
 * href="https://docs.aws.amazon.com/personalize/latest/dg/API_DescribeCampaign.html">DescribeCampaign</a>
 * API.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/personalize-2018-05-22/CampaignUpdateSummary">AWS
 * API Reference</a></p>
 */
class CampaignUpdateSummary {
 public:
  AWS_PERSONALIZE_API CampaignUpdateSummary() = default;
  AWS_PERSONALIZE_API CampaignUpdateSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API CampaignUpdateSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the deployed solution version.</p>
   */
  inline const Aws::String& GetSolutionVersionArn() const { return m_solutionVersionArn; }
  inline bool SolutionVersionArnHasBeenSet() const { return m_solutionVersionArnHasBeenSet; }
  template <typename SolutionVersionArnT = Aws::String>
  void SetSolutionVersionArn(SolutionVersionArnT&& value) {
    m_solutionVersionArnHasBeenSet = true;
    m_solutionVersionArn = std::forward<SolutionVersionArnT>(value);
  }
  template <typename SolutionVersionArnT = Aws::String>
  CampaignUpdateSummary& WithSolutionVersionArn(SolutionVersionArnT&& value) {
    SetSolutionVersionArn(std::forward<SolutionVersionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the requested minimum provisioned transactions (recommendations)
   * per second that Amazon Personalize will support.</p>
   */
  inline int GetMinProvisionedTPS() const { return m_minProvisionedTPS; }
  inline bool MinProvisionedTPSHasBeenSet() const { return m_minProvisionedTPSHasBeenSet; }
  inline void SetMinProvisionedTPS(int value) {
    m_minProvisionedTPSHasBeenSet = true;
    m_minProvisionedTPS = value;
  }
  inline CampaignUpdateSummary& WithMinProvisionedTPS(int value) {
    SetMinProvisionedTPS(value);
    return *this;
  }
  ///@}

  ///@{

  inline const CampaignConfig& GetCampaignConfig() const { return m_campaignConfig; }
  inline bool CampaignConfigHasBeenSet() const { return m_campaignConfigHasBeenSet; }
  template <typename CampaignConfigT = CampaignConfig>
  void SetCampaignConfig(CampaignConfigT&& value) {
    m_campaignConfigHasBeenSet = true;
    m_campaignConfig = std::forward<CampaignConfigT>(value);
  }
  template <typename CampaignConfigT = CampaignConfig>
  CampaignUpdateSummary& WithCampaignConfig(CampaignConfigT&& value) {
    SetCampaignConfig(std::forward<CampaignConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the campaign update.</p> <p>A campaign update can be in one of
   * the following states:</p> <ul> <li> <p>CREATE PENDING &gt; CREATE IN_PROGRESS
   * &gt; ACTIVE -or- CREATE FAILED</p> </li> <li> <p>DELETE PENDING &gt; DELETE
   * IN_PROGRESS</p> </li> </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  CampaignUpdateSummary& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If a campaign update fails, the reason behind the failure.</p>
   */
  inline const Aws::String& GetFailureReason() const { return m_failureReason; }
  inline bool FailureReasonHasBeenSet() const { return m_failureReasonHasBeenSet; }
  template <typename FailureReasonT = Aws::String>
  void SetFailureReason(FailureReasonT&& value) {
    m_failureReasonHasBeenSet = true;
    m_failureReason = std::forward<FailureReasonT>(value);
  }
  template <typename FailureReasonT = Aws::String>
  CampaignUpdateSummary& WithFailureReason(FailureReasonT&& value) {
    SetFailureReason(std::forward<FailureReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time (in Unix time) that the campaign update was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDateTime() const { return m_creationDateTime; }
  inline bool CreationDateTimeHasBeenSet() const { return m_creationDateTimeHasBeenSet; }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  void SetCreationDateTime(CreationDateTimeT&& value) {
    m_creationDateTimeHasBeenSet = true;
    m_creationDateTime = std::forward<CreationDateTimeT>(value);
  }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  CampaignUpdateSummary& WithCreationDateTime(CreationDateTimeT&& value) {
    SetCreationDateTime(std::forward<CreationDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time (in Unix time) that the campaign update was last
   * updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedDateTime() const { return m_lastUpdatedDateTime; }
  inline bool LastUpdatedDateTimeHasBeenSet() const { return m_lastUpdatedDateTimeHasBeenSet; }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  void SetLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    m_lastUpdatedDateTimeHasBeenSet = true;
    m_lastUpdatedDateTime = std::forward<LastUpdatedDateTimeT>(value);
  }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  CampaignUpdateSummary& WithLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    SetLastUpdatedDateTime(std::forward<LastUpdatedDateTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_solutionVersionArn;
  bool m_solutionVersionArnHasBeenSet = false;

  int m_minProvisionedTPS{0};
  bool m_minProvisionedTPSHasBeenSet = false;

  CampaignConfig m_campaignConfig;
  bool m_campaignConfigHasBeenSet = false;

  Aws::String m_status;
  bool m_statusHasBeenSet = false;

  Aws::String m_failureReason;
  bool m_failureReasonHasBeenSet = false;

  Aws::Utils::DateTime m_creationDateTime{};
  bool m_creationDateTimeHasBeenSet = false;

  Aws::Utils::DateTime m_lastUpdatedDateTime{};
  bool m_lastUpdatedDateTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
