﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/PersonalizeRequest.h>
#include <aws/personalize/Personalize_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Personalize {
namespace Model {

/**
 */
class ListBatchInferenceJobsRequest : public PersonalizeRequest {
 public:
  AWS_PERSONALIZE_API ListBatchInferenceJobsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListBatchInferenceJobs"; }

  AWS_PERSONALIZE_API Aws::String SerializePayload() const override;

  AWS_PERSONALIZE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the solution version from which the batch
   * inference jobs were created.</p>
   */
  inline const Aws::String& GetSolutionVersionArn() const { return m_solutionVersionArn; }
  inline bool SolutionVersionArnHasBeenSet() const { return m_solutionVersionArnHasBeenSet; }
  template <typename SolutionVersionArnT = Aws::String>
  void SetSolutionVersionArn(SolutionVersionArnT&& value) {
    m_solutionVersionArnHasBeenSet = true;
    m_solutionVersionArn = std::forward<SolutionVersionArnT>(value);
  }
  template <typename SolutionVersionArnT = Aws::String>
  ListBatchInferenceJobsRequest& WithSolutionVersionArn(SolutionVersionArnT&& value) {
    SetSolutionVersionArn(std::forward<SolutionVersionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to request the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListBatchInferenceJobsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of batch inference job results to return in each page. The
   * default value is 100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListBatchInferenceJobsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_solutionVersionArn;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_solutionVersionArnHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
