/*
 * Decompiled with CFR 0.152.
 */
package org.apache.distributedlog;

import java.io.File;
import java.io.IOException;
import java.net.BindException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.shims.zk.ZooKeeperServerShim;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.bookkeeper.util.LocalBookKeeper;
import org.apache.commons.configuration.CompositeConfiguration;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.distributedlog.impl.metadata.BKDLConfig;
import org.apache.distributedlog.metadata.DLMetadata;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalDLMEmulator {
    private static final Logger LOG = LoggerFactory.getLogger(LocalDLMEmulator.class);
    public static final String DLOG_NAMESPACE = "/messaging/distributedlog";
    private static final int DEFAULT_BOOKIE_INITIAL_PORT = 0;
    private static final int DEFAULT_ZK_TIMEOUT_SEC = 10;
    private static final int DEFAULT_ZK_PORT = 2181;
    private static final String DEFAULT_ZK_HOST = "127.0.0.1";
    private static final String DEFAULT_ZK_ENSEMBLE = "127.0.0.1:2181";
    private static final int DEFAULT_NUM_BOOKIES = 3;
    private static final ServerConfiguration DEFAULT_SERVER_CONFIGURATION = new ServerConfiguration();
    private final String zkEnsemble;
    private final URI uri;
    private final List<File> tmpDirs = new ArrayList<File>();
    private final int zkTimeoutSec;
    private final Thread bkStartupThread;
    private final String zkHost;
    private final int zkPort;
    private final int numBookies;
    private final LocalBookKeeper lb;

    public static Builder newBuilder() {
        return new Builder();
    }

    private LocalDLMEmulator(final int numBookies, boolean shouldStartZK, String zkHost, int zkPort, int zkTimeoutSec, final ServerConfiguration serverConf) throws Exception {
        this.numBookies = numBookies;
        this.zkHost = zkHost;
        this.zkPort = zkPort;
        this.zkEnsemble = zkHost + ":" + zkPort;
        this.uri = URI.create("distributedlog://" + this.zkEnsemble + DLOG_NAMESPACE);
        this.zkTimeoutSec = zkTimeoutSec;
        this.lb = LocalBookKeeper.getLocalBookies((String)zkHost, (int)zkPort, (int)numBookies, (boolean)shouldStartZK, (ServerConfiguration)serverConf);
        this.bkStartupThread = new Thread(){

            @Override
            public void run() {
                try {
                    try {
                        LOG.info("Starting {} bookies : allowLoopback = {}", (Object)numBookies, (Object)serverConf.getAllowLoopback());
                        LocalDLMEmulator.this.lb.start();
                        LOG.info("{} bookies are started.", (Object)numBookies);
                        while (true) {
                            Thread.sleep(1000L);
                        }
                    }
                    catch (Throwable throwable) {
                        LocalDLMEmulator.this.lb.close();
                        throw throwable;
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                catch (Exception e) {
                    LOG.error("Error starting local bk", (Throwable)e);
                }
            }
        };
    }

    public void start() throws Exception {
        this.bkStartupThread.start();
        if (!LocalBookKeeper.waitForServerUp((String)this.zkEnsemble, (long)(this.zkTimeoutSec * 1000))) {
            throw new Exception("Error starting zookeeper/bookkeeper");
        }
        int bookiesUp = this.checkBookiesUp(this.numBookies, this.zkTimeoutSec);
        if (this.numBookies != bookiesUp) {
            LOG.info("Only {} bookies are up, expected {} bookies to be there.", (Object)bookiesUp, (Object)this.numBookies);
        }
        assert (this.numBookies == bookiesUp);
        DLMetadata.create(new BKDLConfig(this.zkEnsemble, "/ledgers")).create(this.uri);
    }

    public void teardown() throws Exception {
        if (this.bkStartupThread != null) {
            this.bkStartupThread.interrupt();
            this.bkStartupThread.join();
        }
        for (File dir : this.tmpDirs) {
            FileUtils.forceDeleteOnExit((File)dir);
        }
    }

    public String getZkServers() {
        return this.zkEnsemble;
    }

    public URI getUri() {
        return this.uri;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int checkBookiesUp(int count, int timeout) throws Exception {
        try (ZooKeeper zkc = LocalDLMEmulator.connectZooKeeper(this.zkHost, this.zkPort, this.zkTimeoutSec);){
            int mostRecentSize = 0;
            for (int i = 0; i < timeout; ++i) {
                try {
                    List children = zkc.getChildren("/ledgers/available", false);
                    children.remove("readonly");
                    mostRecentSize = children.size();
                    if (mostRecentSize > count || LOG.isDebugEnabled()) {
                        LOG.info("Found " + mostRecentSize + " bookies up, waiting for " + count);
                        if (mostRecentSize > count || LOG.isTraceEnabled()) {
                            for (String child : children) {
                                LOG.info(" server: " + child);
                            }
                        }
                    }
                    if (mostRecentSize == count) {
                        break;
                    }
                }
                catch (KeeperException keeperException) {
                    // empty catch block
                }
                Thread.sleep(1000L);
            }
            int n = mostRecentSize;
            return n;
        }
    }

    public void addBookie() throws Exception {
        this.lb.addBookie();
    }

    public void removeBookie() throws Exception {
        this.lb.removeBookie();
    }

    public static String getBkLedgerPath() {
        return "/ledgers";
    }

    public static ZooKeeper connectZooKeeper(String zkHost, int zkPort) throws IOException, KeeperException, InterruptedException {
        return LocalDLMEmulator.connectZooKeeper(zkHost, zkPort, 10);
    }

    public static ZooKeeper connectZooKeeper(String zkHost, int zkPort, int zkTimeoutSec) throws IOException, KeeperException, InterruptedException {
        final CountDownLatch latch = new CountDownLatch(1);
        String zkHostPort = zkHost + ":" + zkPort;
        ZooKeeper zkc = new ZooKeeper(zkHostPort, zkTimeoutSec * 1000, new Watcher(){

            public void process(WatchedEvent event) {
                if (event.getState() == Watcher.Event.KeeperState.SyncConnected) {
                    latch.countDown();
                }
            }
        });
        if (!latch.await(zkTimeoutSec, TimeUnit.SECONDS)) {
            throw new IOException("Zookeeper took too long to connect");
        }
        return zkc;
    }

    public static URI createDLMURI(String path) throws Exception {
        return LocalDLMEmulator.createDLMURI(DEFAULT_ZK_ENSEMBLE, path);
    }

    public static URI createDLMURI(String zkServers, String path) throws Exception {
        return URI.create("distributedlog://" + zkServers + DLOG_NAMESPACE + path);
    }

    public static Pair<ZooKeeperServerShim, Integer> runZookeeperOnAnyPort(File zkDir) throws Exception {
        return LocalDLMEmulator.runZookeeperOnAnyPort((int)(Math.random() * 10000.0 + 7000.0), zkDir);
    }

    public static Pair<ZooKeeperServerShim, Integer> runZookeeperOnAnyPort(int basePort, File zkDir) throws Exception {
        int maxRetries = 20;
        int minPort = 1025;
        int maxPort = 65535;
        ZooKeeperServerShim zks = null;
        int zkPort = basePort;
        boolean success = false;
        int retries = 0;
        while (!success) {
            try {
                LOG.info("zk trying to bind to port " + zkPort);
                zks = LocalBookKeeper.runZookeeper((int)1000, (int)zkPort, (File)zkDir);
                success = true;
            }
            catch (BindException be) {
                if (++retries > 20) {
                    throw be;
                }
                if (++zkPort <= 65535) continue;
                zkPort = 1025;
            }
        }
        return Pair.of(zks, (Object)zkPort);
    }

    public static void main(String[] args) throws Exception {
        try {
            if (args.length < 1) {
                System.out.println("Usage: LocalDLEmulator [<zk_host>] <zk_port>");
                System.exit(-1);
            }
            String zkHost = DEFAULT_ZK_HOST;
            int zkPort = 2181;
            if (args.length == 1) {
                zkPort = Integer.parseInt(args[0]);
            } else {
                zkHost = args[0];
                zkPort = Integer.parseInt(args[1]);
            }
            final File zkDir = IOUtils.createTempDir((String)"distrlog", (String)"zookeeper");
            final LocalDLMEmulator localDlm = LocalDLMEmulator.newBuilder().zkHost(zkHost).zkPort(zkPort).build();
            Runtime.getRuntime().addShutdownHook(new Thread(){

                @Override
                public void run() {
                    try {
                        localDlm.teardown();
                        FileUtils.forceDeleteOnExit((File)zkDir);
                        System.out.println("ByeBye!");
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
            });
            localDlm.start();
            System.out.println(String.format("DistributedLog Sandbox is running now. You could access distributedlog://%s:%s", zkHost, zkPort));
        }
        catch (Exception ex) {
            System.out.println("Exception occurred running emulator " + ex);
        }
    }

    public static class Builder {
        private int zkTimeoutSec = 10;
        private int numBookies = 3;
        private String zkHost = "127.0.0.1";
        private int zkPort = 2181;
        private boolean shouldStartZK = true;
        private Optional<ServerConfiguration> serverConf = Optional.empty();

        public Builder numBookies(int numBookies) {
            this.numBookies = numBookies;
            return this;
        }

        public Builder zkHost(String zkHost) {
            this.zkHost = zkHost;
            return this;
        }

        public Builder zkPort(int zkPort) {
            this.zkPort = zkPort;
            return this;
        }

        public Builder zkTimeoutSec(int zkTimeoutSec) {
            this.zkTimeoutSec = zkTimeoutSec;
            return this;
        }

        public Builder shouldStartZK(boolean shouldStartZK) {
            this.shouldStartZK = shouldStartZK;
            return this;
        }

        public Builder serverConf(ServerConfiguration serverConf) {
            this.serverConf = Optional.of(serverConf);
            return this;
        }

        public LocalDLMEmulator build() throws Exception {
            ServerConfiguration conf = null;
            if (this.serverConf.isPresent()) {
                conf = this.serverConf.get();
            } else {
                conf = (ServerConfiguration)DEFAULT_SERVER_CONFIGURATION.clone();
                conf.setZkTimeout(this.zkTimeoutSec * 1000);
            }
            ServerConfiguration newConf = new ServerConfiguration();
            newConf.loadConf((CompositeConfiguration)conf);
            newConf.setAllowLoopback(true);
            return new LocalDLMEmulator(this.numBookies, this.shouldStartZK, this.zkHost, this.zkPort, this.zkTimeoutSec, newConf);
        }
    }
}

