/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver.metrics;

import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.hbase.HBaseClassTestRule;
import org.apache.hadoop.hbase.metrics.Counter;
import org.apache.hadoop.hbase.metrics.MetricRegistries;
import org.apache.hadoop.hbase.metrics.MetricRegistry;
import org.apache.hadoop.hbase.metrics.MetricRegistryInfo;
import org.apache.hadoop.hbase.quotas.RpcThrottlingException;
import org.apache.hadoop.hbase.regionserver.metrics.MetricsThrottleExceptions;
import org.apache.hadoop.hbase.testclassification.RegionServerTests;
import org.apache.hadoop.hbase.testclassification.SmallTests;
import org.junit.After;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={RegionServerTests.class, SmallTests.class})
public class TestMetricsThrottleExceptions {
    @ClassRule
    public static final HBaseClassTestRule CLASS_RULE = HBaseClassTestRule.forClass(TestMetricsThrottleExceptions.class);
    private MetricRegistry testRegistry;
    private MetricsThrottleExceptions throttleMetrics;

    @After
    public void cleanup() {
        MetricRegistries.global().clear();
    }

    @Test
    public void testBasicThrottleMetricsRecording() {
        this.setupTestMetrics();
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "users");
        Optional metric = this.testRegistry.get("RpcThrottlingException_Type_NumRequestsExceeded_User_alice_Table_users");
        Assert.assertTrue((String)"Counter metric should be present", (boolean)metric.isPresent());
        Assert.assertTrue((String)"Metric should be a counter", (boolean)(metric.get() instanceof Counter));
        Counter counter = (Counter)metric.get();
        Assert.assertEquals((String)"Counter should have count of 1", (long)1L, (long)counter.getCount());
    }

    @Test
    public void testMultipleThrottleTypes() {
        this.setupTestMetrics();
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "users");
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.WriteSizeExceeded, "bob", "logs");
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.ReadSizeExceeded, "charlie", "metadata");
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_NumRequestsExceeded_User_alice_Table_users", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_WriteSizeExceeded_User_bob_Table_logs", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_ReadSizeExceeded_User_charlie_Table_metadata", 1L);
    }

    @Test
    public void testCounterIncrement() {
        this.setupTestMetrics();
        String metricName = "RpcThrottlingException_Type_NumRequestsExceeded_User_alice_Table_users";
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "users");
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "users");
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "users");
        this.verifyCounter(this.testRegistry, metricName, 3L);
    }

    @Test
    public void testMetricNameSanitization() {
        this.setupTestMetrics();
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.WriteSizeExceeded, "user.name@company", "my-table-prod");
        String expectedMetricName = "RpcThrottlingException_Type_WriteSizeExceeded_User_user.name@company_Table_my-table-prod";
        this.verifyCounter(this.testRegistry, expectedMetricName, 1L);
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.ReadSizeExceeded, "user,with=bad:chars*", "table?with\"quotes");
        String problematicMetricName = "RpcThrottlingException_Type_ReadSizeExceeded_User_user_with_bad_chars__Table_table_with_quotes";
        this.verifyCounter(this.testRegistry, problematicMetricName, 1L);
    }

    @Test
    public void testNullHandling() {
        this.setupTestMetrics();
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, null, null);
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.WriteSizeExceeded, "alice", null);
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.ReadSizeExceeded, null, "users");
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_NumRequestsExceeded_User_unknown_Table_unknown", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_WriteSizeExceeded_User_alice_Table_unknown", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_ReadSizeExceeded_User_unknown_Table_users", 1L);
    }

    @Test
    public void testConcurrentAccess() throws InterruptedException {
        this.setupTestMetrics();
        int numThreads = 10;
        int incrementsPerThread = 100;
        ExecutorService executor = Executors.newFixedThreadPool(numThreads);
        CountDownLatch startLatch = new CountDownLatch(1);
        CountDownLatch doneLatch = new CountDownLatch(numThreads);
        AtomicInteger exceptions = new AtomicInteger(0);
        for (int i = 0; i < numThreads; ++i) {
            executor.submit(() -> {
                try {
                    startLatch.await();
                    for (int j = 0; j < incrementsPerThread; ++j) {
                        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "users");
                    }
                }
                catch (Exception e) {
                    exceptions.incrementAndGet();
                }
                finally {
                    doneLatch.countDown();
                }
            });
        }
        startLatch.countDown();
        boolean completed = doneLatch.await(30L, TimeUnit.SECONDS);
        Assert.assertTrue((String)"All threads should complete within timeout", (boolean)completed);
        Assert.assertEquals((String)"No exceptions should occur during concurrent access", (long)0L, (long)exceptions.get());
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_NumRequestsExceeded_User_alice_Table_users", numThreads * incrementsPerThread);
        executor.shutdown();
    }

    @Test
    public void testCommonTableNamePatterns() {
        this.setupTestMetrics();
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "service-user", "my-app-logs");
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.WriteSizeExceeded, "batch.process", "namespace:table-name");
        this.throttleMetrics.recordThrottleException(RpcThrottlingException.Type.ReadSizeExceeded, "user_123", "test_table_v2");
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_NumRequestsExceeded_User_service-user_Table_my-app-logs", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_WriteSizeExceeded_User_batch.process_Table_namespace_table-name", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_ReadSizeExceeded_User_user_123_Table_test_table_v2", 1L);
    }

    @Test
    public void testAllThrottleExceptionTypes() {
        RpcThrottlingException.Type[] throttleTypes;
        this.setupTestMetrics();
        for (RpcThrottlingException.Type throttleType : throttleTypes = RpcThrottlingException.Type.values()) {
            this.throttleMetrics.recordThrottleException(throttleType, "testuser", "testtable");
        }
        for (RpcThrottlingException.Type throttleType : throttleTypes) {
            String expectedMetricName = "RpcThrottlingException_Type_" + throttleType.name() + "_User_testuser_Table_testtable";
            this.verifyCounter(this.testRegistry, expectedMetricName, 1L);
        }
    }

    @Test
    public void testMultipleInstances() {
        this.setupTestMetrics();
        MetricsThrottleExceptions metrics1 = new MetricsThrottleExceptions(this.testRegistry);
        MetricsThrottleExceptions metrics2 = new MetricsThrottleExceptions(this.testRegistry);
        metrics1.recordThrottleException(RpcThrottlingException.Type.NumRequestsExceeded, "alice", "table1");
        metrics2.recordThrottleException(RpcThrottlingException.Type.WriteSizeExceeded, "bob", "table2");
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_NumRequestsExceeded_User_alice_Table_table1", 1L);
        this.verifyCounter(this.testRegistry, "RpcThrottlingException_Type_WriteSizeExceeded_User_bob_Table_table2", 1L);
    }

    private void setupTestMetrics() {
        MetricRegistryInfo registryInfo = this.getRegistryInfo();
        this.testRegistry = MetricRegistries.global().create(registryInfo);
        this.throttleMetrics = new MetricsThrottleExceptions(this.testRegistry);
    }

    private void verifyCounter(MetricRegistry registry, String metricName, long expectedCount) {
        Optional metric = registry.get(metricName);
        Assert.assertTrue((String)("Counter metric '" + metricName + "' should be present"), (boolean)metric.isPresent());
        Assert.assertTrue((String)"Metric should be a counter", (boolean)(metric.get() instanceof Counter));
        Counter counter = (Counter)metric.get();
        Assert.assertEquals((String)("Counter '" + metricName + "' should have expected count"), (long)expectedCount, (long)counter.getCount());
    }

    private MetricRegistryInfo getRegistryInfo() {
        return new MetricRegistryInfo("ThrottleExceptions", "Metrics about RPC throttling exceptions", "RegionServer,sub=ThrottleExceptions", "regionserver", false);
    }
}

