/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseClassTestRule;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.io.ByteBuffAllocator;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheTestUtils;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocator;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocatorException;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCache;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketEntry;
import org.apache.hadoop.hbase.io.hfile.bucket.CacheFullException;
import org.apache.hadoop.hbase.io.hfile.bucket.IOEngine;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HStore;
import org.apache.hadoop.hbase.regionserver.HStoreFile;
import org.apache.hadoop.hbase.testclassification.IOTests;
import org.apache.hadoop.hbase.testclassification.LargeTests;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableMap;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.Mockito;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@RunWith(value=Parameterized.class)
@Category(value={IOTests.class, LargeTests.class})
public class TestBucketCache {
    private static final Logger LOG = LoggerFactory.getLogger(TestBucketCache.class);
    @ClassRule
    public static final HBaseClassTestRule CLASS_RULE = HBaseClassTestRule.forClass(TestBucketCache.class);
    @Parameterized.Parameter(value=0)
    public int constructedBlockSize;
    @Parameterized.Parameter(value=1)
    public int[] constructedBlockSizes;
    BucketCache cache;
    final int CACHE_SIZE = 1000000;
    final int NUM_BLOCKS = 100;
    final int BLOCK_SIZE = 10000;
    final int NUM_THREADS = 100;
    final int NUM_QUERIES = 10000;
    final long capacitySize = 0x2000000L;
    final int writeThreads = 3;
    final int writerQLen = 64;
    private String ioEngineName = "offheap";
    private static final HBaseTestingUtility HBASE_TESTING_UTILITY = new HBaseTestingUtility();

    @Parameterized.Parameters(name="{index}: blockSize={0}, bucketSizes={1}")
    public static Iterable<Object[]> data() {
        return Arrays.asList({8192, null}, {16384, new int[]{3072, 5120, 9216, 17408, 29696, 33792, 66560, 99328, 132096}});
    }

    @Before
    public void setup() throws IOException {
        this.cache = new MockedBucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, null);
    }

    @After
    public void tearDown() {
        this.cache.shutdown();
    }

    private Path createAndGetTestDir() throws IOException {
        Path testDir = HBASE_TESTING_UTILITY.getDataTestDir();
        HBASE_TESTING_UTILITY.getTestFileSystem().mkdirs(testDir);
        return testDir;
    }

    private static <T> T randFrom(List<T> a) {
        return a.get(ThreadLocalRandom.current().nextInt(a.size()));
    }

    @Test
    public void testBucketAllocator() throws BucketAllocatorException {
        BucketAllocator mAllocator = this.cache.getAllocator();
        List<Integer> BLOCKSIZES = Arrays.asList(4096, 8192, 65536, 98304);
        boolean full = false;
        ArrayList<Pair> allocations = new ArrayList<Pair>();
        ArrayList<Integer> tmp = new ArrayList<Integer>(BLOCKSIZES);
        while (!full) {
            Integer blockSize = null;
            try {
                blockSize = TestBucketCache.randFrom(tmp);
                allocations.add(new Pair((Object)mAllocator.allocateBlock(blockSize.intValue()), (Object)blockSize));
            }
            catch (CacheFullException cfe) {
                tmp.remove(blockSize);
                if (!tmp.isEmpty()) continue;
                full = true;
            }
        }
        for (Integer blockSize : BLOCKSIZES) {
            BucketAllocator.BucketSizeInfo bucketSizeInfo = mAllocator.roundUpToBucketSizeInfo(blockSize.intValue());
            BucketAllocator.IndexStatistics indexStatistics = bucketSizeInfo.statistics();
            Assert.assertEquals((String)("unexpected freeCount for " + bucketSizeInfo), (long)0L, (long)indexStatistics.freeCount());
            Assert.assertEquals((long)(1024L * indexStatistics.totalCount()), (long)indexStatistics.fragmentationBytes());
        }
        mAllocator.logDebugStatistics();
        for (Pair allocation : allocations) {
            Assert.assertEquals((long)mAllocator.sizeOfAllocation(((Long)allocation.getFirst()).longValue()), (long)mAllocator.freeBlock(((Long)allocation.getFirst()).longValue(), ((Integer)allocation.getSecond()).intValue()));
        }
        Assert.assertEquals((long)0L, (long)mAllocator.getUsedSize());
    }

    @Test
    public void testCacheSimple() throws Exception {
        CacheTestUtils.testCacheSimple((BlockCache)this.cache, 10000, 10000);
    }

    @Test
    public void testCacheMultiThreadedSingleKey() throws Exception {
        CacheTestUtils.hammerSingleKey((BlockCache)this.cache, 200, 20000);
    }

    @Test
    public void testHeapSizeChanges() throws Exception {
        this.cache.stopWriterThreads();
        CacheTestUtils.testHeapSizeChanges((BlockCache)this.cache, 10000);
    }

    public static void waitUntilFlushedToBucket(BucketCache cache, BlockCacheKey cacheKey) throws InterruptedException {
        while (!cache.backingMap.containsKey(cacheKey) || cache.ramCache.containsKey(cacheKey)) {
            Thread.sleep(100L);
        }
        Thread.sleep(1000L);
    }

    public static void waitUntilAllFlushedToBucket(BucketCache cache) throws InterruptedException {
        while (!cache.ramCache.isEmpty()) {
            Thread.sleep(100L);
        }
        Thread.sleep(1000L);
    }

    private void cacheAndWaitUntilFlushedToBucket(BucketCache cache, BlockCacheKey cacheKey, Cacheable block, boolean waitWhenCache) throws InterruptedException {
        cache.cacheBlock(cacheKey, block, false, waitWhenCache);
        TestBucketCache.waitUntilFlushedToBucket(cache, cacheKey);
    }

    @Test
    public void testMemoryLeak() throws Exception {
        final BlockCacheKey cacheKey = new BlockCacheKey("dummy", 1L);
        this.cacheAndWaitUntilFlushedToBucket(this.cache, cacheKey, new CacheTestUtils.ByteArrayCacheable(new byte[10]), true);
        long lockId = ((BucketEntry)this.cache.backingMap.get(cacheKey)).offset();
        ReentrantReadWriteLock lock = this.cache.offsetLock.getLock((Object)lockId);
        lock.writeLock().lock();
        Thread evictThread = new Thread("evict-block"){

            @Override
            public void run() {
                TestBucketCache.this.cache.evictBlock(cacheKey);
            }
        };
        evictThread.start();
        this.cache.offsetLock.waitForWaiters((Object)lockId, 1);
        this.cache.blockEvicted(cacheKey, (BucketEntry)this.cache.backingMap.remove(cacheKey), true, true);
        Assert.assertEquals((long)0L, (long)this.cache.getBlockCount());
        this.cacheAndWaitUntilFlushedToBucket(this.cache, cacheKey, new CacheTestUtils.ByteArrayCacheable(new byte[10]), true);
        Assert.assertEquals((long)1L, (long)this.cache.getBlockCount());
        lock.writeLock().unlock();
        evictThread.join();
        Assert.assertEquals((long)1L, (long)this.cache.getBlockCount());
        Assert.assertTrue((this.cache.getCurrentSize() > 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)"We should have a block!", (boolean)this.cache.iterator().hasNext());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testRetrieveFromFile() throws Exception {
        Path testDir = this.createAndGetTestDir();
        String ioEngineName = "file:" + testDir + "/bucket.cache";
        this.testRetrievalUtils(testDir, ioEngineName);
        int[] smallBucketSizes = new int[]{3072, 5120};
        String persistencePath = testDir + "/bucket.persistence";
        BucketCache bucketCache = null;
        try {
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, smallBucketSizes, 3, 64, persistencePath);
            Assert.assertFalse((boolean)new File(persistencePath).exists());
            Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
            Assert.assertEquals((long)0L, (long)bucketCache.backingMap.size());
        }
        finally {
            bucketCache.shutdown();
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    @Test
    public void testRetrieveFromMMap() throws Exception {
        Path testDir = this.createAndGetTestDir();
        String ioEngineName = "mmap:" + testDir + "/bucket.cache";
        this.testRetrievalUtils(testDir, ioEngineName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testRetrieveFromPMem() throws Exception {
        Path testDir = this.createAndGetTestDir();
        String ioEngineName = "pmem:" + testDir + "/bucket.cache";
        this.testRetrievalUtils(testDir, ioEngineName);
        int[] smallBucketSizes = new int[]{3072, 5120};
        String persistencePath = testDir + "/bucket.persistence" + EnvironmentEdgeManager.currentTime();
        BucketCache bucketCache = null;
        try {
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, smallBucketSizes, 3, 64, persistencePath);
            Assert.assertFalse((boolean)new File(persistencePath).exists());
            Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
            Assert.assertEquals((long)0L, (long)bucketCache.backingMap.size());
        }
        finally {
            bucketCache.shutdown();
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testRetrievalUtils(Path testDir, String ioEngineName) throws IOException, InterruptedException {
        String persistencePath = testDir + "/bucket.persistence" + EnvironmentEdgeManager.currentTime();
        BucketCache bucketCache = null;
        try {
            CacheTestUtils.HFileBlockPair[] blocks;
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, persistencePath);
            long usedSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertEquals((long)0L, (long)usedSize);
            for (CacheTestUtils.HFileBlockPair block : blocks = CacheTestUtils.generateHFileBlocks(this.constructedBlockSize, 1)) {
                bucketCache.cacheBlock(block.getBlockName(), (Cacheable)block.getBlock());
            }
            for (CacheTestUtils.HFileBlockPair block : blocks) {
                this.cacheAndWaitUntilFlushedToBucket(bucketCache, block.getBlockName(), (Cacheable)block.getBlock(), false);
            }
            usedSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertNotEquals((long)0L, (long)usedSize);
            bucketCache.shutdown();
            Assert.assertTrue((boolean)new File(persistencePath).exists());
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, persistencePath);
            Assert.assertEquals((long)usedSize, (long)bucketCache.getAllocator().getUsedSize());
        }
        finally {
            if (bucketCache != null) {
                bucketCache.shutdown();
            }
        }
        Assert.assertTrue((boolean)new File(persistencePath).exists());
    }

    @Test
    public void testRetrieveUnsupportedIOE() throws Exception {
        try {
            Path testDir = this.createAndGetTestDir();
            String ioEngineName = testDir + "/bucket.cache";
            this.testRetrievalUtils(testDir, ioEngineName);
            Assert.fail((String)"Should have thrown IllegalArgumentException because of unsupported IOEngine!!");
        }
        catch (IllegalArgumentException e) {
            Assert.assertEquals((Object)"Don't understand io engine name for cache- prefix with file:, files:, mmap: or offheap", (Object)e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testRetrieveFromMultipleFiles() throws Exception {
        Path testDirInitial = this.createAndGetTestDir();
        Path newTestDir = new HBaseTestingUtility().getDataTestDir();
        HBASE_TESTING_UTILITY.getTestFileSystem().mkdirs(newTestDir);
        String ioEngineName = "files:" + testDirInitial + "/bucket1.cache" + "," + newTestDir + "/bucket2.cache";
        this.testRetrievalUtils(testDirInitial, ioEngineName);
        int[] smallBucketSizes = new int[]{3072, 5120};
        String persistencePath = testDirInitial + "/bucket.persistence";
        BucketCache bucketCache = null;
        try {
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, smallBucketSizes, 3, 64, persistencePath);
            Assert.assertFalse((boolean)new File(persistencePath).exists());
            Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
            Assert.assertEquals((long)0L, (long)bucketCache.backingMap.size());
        }
        finally {
            bucketCache.shutdown();
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testRetrieveFromFileWithoutPersistence() throws Exception {
        BucketCache bucketCache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, null);
        try {
            CacheTestUtils.HFileBlockPair[] blocks;
            Path testDir = this.createAndGetTestDir();
            String ioEngineName = "file:" + testDir + "/bucket.cache";
            long usedSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertEquals((long)0L, (long)usedSize);
            for (CacheTestUtils.HFileBlockPair block : blocks = CacheTestUtils.generateHFileBlocks(this.constructedBlockSize, 1)) {
                bucketCache.cacheBlock(block.getBlockName(), (Cacheable)block.getBlock());
            }
            for (CacheTestUtils.HFileBlockPair block : blocks) {
                this.cacheAndWaitUntilFlushedToBucket(bucketCache, block.getBlockName(), (Cacheable)block.getBlock(), false);
            }
            usedSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertNotEquals((long)0L, (long)usedSize);
            bucketCache.shutdown();
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, null);
            Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
        }
        finally {
            bucketCache.shutdown();
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    @Test
    public void testBucketAllocatorLargeBuckets() throws BucketAllocatorException {
        long availableSpace = 0x500000000L;
        int[] bucketSizes = new int[]{1024, 0x100000, 0x40000000};
        BucketAllocator allocator = new BucketAllocator(availableSpace, bucketSizes);
        Assert.assertTrue((allocator.getBuckets().length > 0 ? 1 : 0) != 0);
    }

    @Test
    public void testGetPartitionSize() throws IOException {
        this.validateGetPartitionSize(this.cache, 0.25f, 0.85f);
        Configuration conf = HBaseConfiguration.create();
        conf.setFloat("hbase.bucketcache.minfactor", 0.5f);
        conf.setFloat("hbase.bucketcache.single.factor", 0.1f);
        conf.setFloat("hbase.bucketcache.multi.factor", 0.7f);
        conf.setFloat("hbase.bucketcache.memory.factor", 0.2f);
        BucketCache cache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, null, 100, conf);
        this.validateGetPartitionSize(cache, 0.1f, 0.5f);
        this.validateGetPartitionSize(cache, 0.7f, 0.5f);
        this.validateGetPartitionSize(cache, 0.2f, 0.5f);
    }

    @Test
    public void testCacheSizeCapacity() throws IOException {
        this.validateGetPartitionSize(this.cache, 0.25f, 0.85f);
        Configuration conf = HBaseConfiguration.create();
        conf.setFloat("hbase.bucketcache.minfactor", 0.5f);
        conf.setFloat("hbase.bucketcache.single.factor", 0.1f);
        conf.setFloat("hbase.bucketcache.multi.factor", 0.7f);
        conf.setFloat("hbase.bucketcache.memory.factor", 0.2f);
        try {
            new BucketCache(this.ioEngineName, Long.MAX_VALUE, 1, this.constructedBlockSizes, 3, 64, null, 100, conf);
            Assert.fail((String)"Should have thrown IllegalArgumentException because of large cache capacity!");
        }
        catch (IllegalArgumentException e) {
            Assert.assertEquals((Object)"Cache capacity is too large, only support 32TB now", (Object)e.getMessage());
        }
    }

    @Test
    public void testValidBucketCacheConfigs() throws IOException {
        Configuration conf = HBaseConfiguration.create();
        conf.setFloat("hbase.bucketcache.acceptfactor", 0.9f);
        conf.setFloat("hbase.bucketcache.minfactor", 0.5f);
        conf.setFloat("hbase.bucketcache.extrafreefactor", 0.5f);
        conf.setFloat("hbase.bucketcache.single.factor", 0.1f);
        conf.setFloat("hbase.bucketcache.multi.factor", 0.7f);
        conf.setFloat("hbase.bucketcache.memory.factor", 0.2f);
        BucketCache cache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, null, 100, conf);
        Assert.assertEquals((String)"hbase.bucketcache.acceptfactor failed to propagate.", (float)0.9f, (float)cache.getAcceptableFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.minfactor failed to propagate.", (float)0.5f, (float)cache.getMinFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.extrafreefactor failed to propagate.", (float)0.5f, (float)cache.getExtraFreeFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.single.factor failed to propagate.", (float)0.1f, (float)cache.getSingleFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.multi.factor failed to propagate.", (float)0.7f, (float)cache.getMultiFactor(), (float)0.0f);
        Assert.assertEquals((String)"hbase.bucketcache.memory.factor failed to propagate.", (float)0.2f, (float)cache.getMemoryFactor(), (float)0.0f);
    }

    @Test
    public void testInvalidAcceptFactorConfig() throws IOException {
        float[] configValues = new float[]{-1.0f, 0.2f, 0.86f, 1.05f};
        boolean[] expectedOutcomes = new boolean[]{false, false, true, false};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.acceptfactor", (Object)configValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    @Test
    public void testInvalidMinFactorConfig() throws IOException {
        float[] configValues = new float[]{-1.0f, 0.0f, 0.96f, 1.05f};
        boolean[] expectedOutcomes = new boolean[]{false, true, false, false};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.minfactor", (Object)configValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    @Test
    public void testInvalidExtraFreeFactorConfig() throws IOException {
        float[] configValues = new float[]{-1.0f, 0.0f, 0.2f, 1.05f};
        boolean[] expectedOutcomes = new boolean[]{false, true, true, true};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.extrafreefactor", (Object)configValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    @Test
    public void testInvalidCacheSplitFactorConfig() throws IOException {
        float[] singleFactorConfigValues = new float[]{0.2f, 0.0f, -0.2f, 1.0f};
        float[] multiFactorConfigValues = new float[]{0.4f, 0.0f, 1.0f, 0.05f};
        float[] memoryFactorConfigValues = new float[]{0.4f, 0.0f, 0.2f, 0.5f};
        boolean[] expectedOutcomes = new boolean[]{true, false, false, false};
        ImmutableMap configMappings = ImmutableMap.of((Object)"hbase.bucketcache.single.factor", (Object)singleFactorConfigValues, (Object)"hbase.bucketcache.multi.factor", (Object)multiFactorConfigValues, (Object)"hbase.bucketcache.memory.factor", (Object)memoryFactorConfigValues);
        Configuration conf = HBaseConfiguration.create();
        this.checkConfigValues(conf, (Map<String, float[]>)configMappings, expectedOutcomes);
    }

    private void checkConfigValues(Configuration conf, Map<String, float[]> configMap, boolean[] expectSuccess) throws IOException {
        Set<String> configNames = configMap.keySet();
        for (int i = 0; i < expectSuccess.length; ++i) {
            try {
                for (String configName : configNames) {
                    conf.setFloat(configName, configMap.get(configName)[i]);
                }
                BucketCache cache = new BucketCache(this.ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, null, 100, conf);
                Assert.assertTrue((String)("Created BucketCache and expected it to succeed: " + expectSuccess[i] + ", but it actually was: " + !expectSuccess[i]), (boolean)expectSuccess[i]);
                continue;
            }
            catch (IllegalArgumentException e) {
                Assert.assertFalse((String)("Created BucketCache and expected it to succeed: " + expectSuccess[i] + ", but it actually was: " + !expectSuccess[i]), (boolean)expectSuccess[i]);
            }
        }
    }

    private void validateGetPartitionSize(BucketCache bucketCache, float partitionFactor, float minFactor) {
        long expectedOutput = (long)Math.floor((float)bucketCache.getAllocator().getTotalSize() * partitionFactor * minFactor);
        Assert.assertEquals((long)expectedOutput, (long)bucketCache.getPartitionSize(partitionFactor));
    }

    @Test
    public void testOffsetProducesPositiveOutput() {
        long testValue = 549888460800L;
        BucketEntry bucketEntry = new BucketEntry(testValue, 10, 10, 10L, true, entry -> ByteBuffAllocator.NONE, ByteBuffAllocator.HEAP);
        Assert.assertEquals((long)testValue, (long)bucketEntry.offset());
    }

    @Test
    public void testEvictionCount() throws InterruptedException {
        int size = 100;
        int length = 33 + size;
        ByteBuffer buf1 = ByteBuffer.allocate(size);
        ByteBuffer buf2 = ByteBuffer.allocate(size);
        HFileContext meta = new HFileContextBuilder().build();
        ByteBuffAllocator allocator = ByteBuffAllocator.HEAP;
        HFileBlock blockWithNextBlockMetadata = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf1), true, -1L, 52, -1, meta, allocator);
        HFileBlock blockWithoutNextBlockMetadata = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf2), true, -1L, -1, -1, meta, allocator);
        BlockCacheKey key = new BlockCacheKey("testEvictionCount", 0L);
        ByteBuffer actualBuffer = ByteBuffer.allocate(length);
        ByteBuffer block1Buffer = ByteBuffer.allocate(length);
        ByteBuffer block2Buffer = ByteBuffer.allocate(length);
        blockWithNextBlockMetadata.serialize(block1Buffer, true);
        blockWithoutNextBlockMetadata.serialize(block2Buffer, true);
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
        TestBucketCache.waitUntilFlushedToBucket(this.cache, key);
        Assert.assertEquals((long)0L, (long)this.cache.getStats().getEvictionCount());
        Assert.assertEquals((long)1L, (long)this.cache.evictBlocksByHfileName("testEvictionCount"));
        Assert.assertEquals((long)0L, (long)this.cache.getStats().getEvictionCount());
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
        TestBucketCache.waitUntilFlushedToBucket(this.cache, key);
        Assert.assertTrue((boolean)this.cache.evictBlock(key));
        Assert.assertEquals((long)0L, (long)this.cache.getStats().getEvictionCount());
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
        TestBucketCache.waitUntilFlushedToBucket(this.cache, key);
        this.cache.freeSpace("testing");
        Assert.assertEquals((long)1L, (long)this.cache.getStats().getEvictionCount());
    }

    @Test
    public void testCacheBlockNextBlockMetadataMissing() throws Exception {
        int size = 100;
        int length = 33 + size;
        ByteBuffer buf1 = ByteBuffer.allocate(size);
        ByteBuffer buf2 = ByteBuffer.allocate(size);
        HFileContext meta = new HFileContextBuilder().build();
        ByteBuffAllocator allocator = ByteBuffAllocator.HEAP;
        HFileBlock blockWithNextBlockMetadata = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf1), true, -1L, 52, -1, meta, allocator);
        HFileBlock blockWithoutNextBlockMetadata = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf2), true, -1L, -1, -1, meta, allocator);
        BlockCacheKey key = new BlockCacheKey("testCacheBlockNextBlockMetadataMissing", 0L);
        ByteBuffer actualBuffer = ByteBuffer.allocate(length);
        ByteBuffer block1Buffer = ByteBuffer.allocate(length);
        ByteBuffer block2Buffer = ByteBuffer.allocate(length);
        blockWithNextBlockMetadata.serialize(block1Buffer, true);
        blockWithoutNextBlockMetadata.serialize(block2Buffer, true);
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
        TestBucketCache.waitUntilFlushedToBucket(this.cache, key);
        Assert.assertNotNull(this.cache.backingMap.get(key));
        Assert.assertEquals((long)1L, (long)((BucketEntry)this.cache.backingMap.get(key)).refCnt());
        Assert.assertEquals((long)1L, (long)blockWithNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)blockWithoutNextBlockMetadata.getBufferReadOnly().refCnt());
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithoutNextBlockMetadata, actualBuffer, block1Buffer);
        Assert.assertEquals((long)1L, (long)blockWithNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)blockWithoutNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)((BucketEntry)this.cache.backingMap.get(key)).refCnt());
        Assert.assertTrue((boolean)this.cache.evictBlock(key));
        Assert.assertEquals((long)1L, (long)blockWithNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)blockWithoutNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertNull((Object)this.cache.getBlock(key, false, false, false));
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithoutNextBlockMetadata, actualBuffer, block2Buffer);
        TestBucketCache.waitUntilFlushedToBucket(this.cache, key);
        Assert.assertEquals((long)1L, (long)blockWithNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)blockWithoutNextBlockMetadata.getBufferReadOnly().refCnt());
        CacheTestUtils.getBlockAndAssertEquals((BlockCache)this.cache, key, (Cacheable)blockWithNextBlockMetadata, actualBuffer, block1Buffer);
        TestBucketCache.waitUntilFlushedToBucket(this.cache, key);
        Assert.assertEquals((long)1L, (long)blockWithNextBlockMetadata.getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)blockWithoutNextBlockMetadata.getBufferReadOnly().refCnt());
    }

    @Test
    public void testRAMCache() {
        int size = 100;
        int length = 33 + size;
        byte[] byteArr = new byte[length];
        ByteBuffer buf = ByteBuffer.wrap(byteArr, 0, size);
        HFileContext meta = new HFileContextBuilder().build();
        BucketCache.RAMCache cache = new BucketCache.RAMCache();
        BlockCacheKey key1 = new BlockCacheKey("file-1", 1L);
        BlockCacheKey key2 = new BlockCacheKey("file-2", 2L);
        HFileBlock blk1 = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf), true, -1L, 52, -1, meta, ByteBuffAllocator.HEAP);
        HFileBlock blk2 = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf), true, -1L, -1, -1, meta, ByteBuffAllocator.HEAP);
        BucketCache.RAMQueueEntry re1 = new BucketCache.RAMQueueEntry(key1, (Cacheable)blk1, 1L, false, false);
        BucketCache.RAMQueueEntry re2 = new BucketCache.RAMQueueEntry(key1, (Cacheable)blk2, 1L, false, false);
        Assert.assertFalse((boolean)cache.containsKey(key1));
        Assert.assertNull((Object)cache.putIfAbsent(key1, re1));
        Assert.assertEquals((long)2L, (long)((HFileBlock)re1.getData()).getBufferReadOnly().refCnt());
        Assert.assertNotNull((Object)cache.putIfAbsent(key1, re2));
        Assert.assertEquals((long)2L, (long)((HFileBlock)re1.getData()).getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)((HFileBlock)re2.getData()).getBufferReadOnly().refCnt());
        Assert.assertNull((Object)cache.putIfAbsent(key2, re2));
        Assert.assertEquals((long)2L, (long)((HFileBlock)re1.getData()).getBufferReadOnly().refCnt());
        Assert.assertEquals((long)2L, (long)((HFileBlock)re2.getData()).getBufferReadOnly().refCnt());
        cache.remove(key1);
        Assert.assertEquals((long)1L, (long)((HFileBlock)re1.getData()).getBufferReadOnly().refCnt());
        Assert.assertEquals((long)2L, (long)((HFileBlock)re2.getData()).getBufferReadOnly().refCnt());
        cache.clear();
        Assert.assertEquals((long)1L, (long)((HFileBlock)re1.getData()).getBufferReadOnly().refCnt());
        Assert.assertEquals((long)1L, (long)((HFileBlock)re2.getData()).getBufferReadOnly().refCnt());
    }

    @Test
    public void testFreeBlockWhenIOEngineWriteFailure() throws IOException {
        int size = 100;
        int offset = 20;
        int length = 33 + size;
        ByteBuffer buf = ByteBuffer.allocate(length);
        HFileContext meta = new HFileContextBuilder().build();
        HFileBlock block = new HFileBlock(BlockType.DATA, size, size, -1L, ByteBuff.wrap((ByteBuffer)buf), true, (long)offset, 52, -1, meta, ByteBuffAllocator.HEAP);
        IOEngine ioEngine = (IOEngine)Mockito.mock(IOEngine.class);
        Mockito.when((Object)ioEngine.usesSharedMemory()).thenReturn((Object)false);
        ((IOEngine)Mockito.doThrow(RuntimeException.class).when((Object)ioEngine)).write((ByteBuffer)Mockito.any(ByteBuffer.class), Mockito.anyLong());
        ((IOEngine)Mockito.doThrow(RuntimeException.class).when((Object)ioEngine)).write((ByteBuff)Mockito.any(ByteBuff.class), Mockito.anyLong());
        long availableSpace = 0x40000000L;
        BucketAllocator allocator = new BucketAllocator(availableSpace, null);
        BlockCacheKey key = new BlockCacheKey("dummy", 1L);
        BucketCache.RAMQueueEntry re = new BucketCache.RAMQueueEntry(key, (Cacheable)block, 1L, true, false);
        Assert.assertEquals((long)0L, (long)allocator.getUsedSize());
        try {
            re.writeToCache(ioEngine, allocator, null, null, ByteBuffer.allocate(13));
            Assert.fail();
        }
        catch (Exception exception) {
            // empty catch block
        }
        Assert.assertEquals((long)0L, (long)allocator.getUsedSize());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFreeBucketEntryRestoredFromFile() throws Exception {
        BucketCache bucketCache = null;
        try {
            CacheTestUtils.HFileBlockPair[] hfileBlockPairs;
            Path dataTestDir = this.createAndGetTestDir();
            String ioEngineName = "file:" + dataTestDir + "/bucketNoRecycler.cache";
            String persistencePath = dataTestDir + "/bucketNoRecycler.persistence";
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, persistencePath);
            long usedByteSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertEquals((long)0L, (long)usedByteSize);
            for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs = CacheTestUtils.generateHFileBlocks(this.constructedBlockSize, 1)) {
                bucketCache.cacheBlock(hfileBlockPair.getBlockName(), (Cacheable)hfileBlockPair.getBlock());
            }
            for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs) {
                this.cacheAndWaitUntilFlushedToBucket(bucketCache, hfileBlockPair.getBlockName(), (Cacheable)hfileBlockPair.getBlock(), false);
            }
            usedByteSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertNotEquals((long)0L, (long)usedByteSize);
            bucketCache.shutdown();
            Assert.assertTrue((boolean)new File(persistencePath).exists());
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, persistencePath);
            Assert.assertEquals((long)usedByteSize, (long)bucketCache.getAllocator().getUsedSize());
            for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs) {
                BlockCacheKey blockCacheKey = hfileBlockPair.getBlockName();
                bucketCache.evictBlock(blockCacheKey);
            }
            Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
            Assert.assertEquals((long)0L, (long)bucketCache.backingMap.size());
        }
        finally {
            bucketCache.shutdown();
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testBlockAdditionWaitWhenCache() throws Exception {
        BucketCache bucketCache = null;
        try {
            CacheTestUtils.HFileBlockPair[] hfileBlockPairs;
            Path dataTestDir = this.createAndGetTestDir();
            String ioEngineName = "file:" + dataTestDir + "/bucketNoRecycler.cache";
            String persistencePath = dataTestDir + "/bucketNoRecycler.persistence";
            Configuration config = HBASE_TESTING_UTILITY.getConfiguration();
            config.setLong("hbase.bucketcache.queue.addition.waittime", 1000L);
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 1, 1, persistencePath);
            long usedByteSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertEquals((long)0L, (long)usedByteSize);
            for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs = CacheTestUtils.generateHFileBlocks(this.constructedBlockSize, 10)) {
                bucketCache.cacheBlock(hfileBlockPair.getBlockName(), (Cacheable)hfileBlockPair.getBlock(), false, true);
            }
            for (long timeout = 10000L; bucketCache.backingMap.size() != 10 && timeout > 0L; timeout -= 100L) {
                Threads.sleep((long)100L);
            }
            for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs) {
                Assert.assertTrue((boolean)bucketCache.backingMap.containsKey(hfileBlockPair.getBlockName()));
            }
            usedByteSize = bucketCache.getAllocator().getUsedSize();
            Assert.assertNotEquals((long)0L, (long)usedByteSize);
            bucketCache.shutdown();
            Assert.assertTrue((boolean)new File(persistencePath).exists());
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 3, 64, persistencePath);
            Assert.assertEquals((long)usedByteSize, (long)bucketCache.getAllocator().getUsedSize());
            for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs) {
                BlockCacheKey blockCacheKey = hfileBlockPair.getBlockName();
                bucketCache.evictBlock(blockCacheKey);
            }
            Assert.assertEquals((long)0L, (long)bucketCache.getAllocator().getUsedSize());
            Assert.assertEquals((long)0L, (long)bucketCache.backingMap.size());
        }
        finally {
            if (bucketCache != null) {
                bucketCache.shutdown();
            }
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testOnConfigurationChange() throws Exception {
        BucketCache bucketCache = null;
        try {
            Path dataTestDir = this.createAndGetTestDir();
            String ioEngineName = "file:" + dataTestDir + "/bucketNoRecycler.cache";
            Configuration config = HBASE_TESTING_UTILITY.getConfiguration();
            bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 1, 1, null, 60000, config);
            config.setFloat("hbase.bucketcache.acceptfactor", 0.9f);
            config.setFloat("hbase.bucketcache.minfactor", 0.8f);
            config.setFloat("hbase.bucketcache.extrafreefactor", 0.15f);
            config.setFloat("hbase.bucketcache.single.factor", 0.2f);
            config.setFloat("hbase.bucketcache.multi.factor", 0.6f);
            config.setFloat("hbase.bucketcache.memory.factor", 0.2f);
            config.setLong("hbase.bucketcache.queue.addition.waittime", 100L);
            config.setLong("hbase.bucketcache.persist.intervalinmillis", 500L);
            config.setLong("hbase.bucketcache.persistence.chunksize", 1000L);
            bucketCache.onConfigurationChange(config);
            Assert.assertEquals((double)0.9f, (double)bucketCache.getAcceptableFactor(), (double)0.01);
            Assert.assertEquals((double)0.8f, (double)bucketCache.getMinFactor(), (double)0.01);
            Assert.assertEquals((double)0.15f, (double)bucketCache.getExtraFreeFactor(), (double)0.01);
            Assert.assertEquals((double)0.2f, (double)bucketCache.getSingleFactor(), (double)0.01);
            Assert.assertEquals((double)0.6f, (double)bucketCache.getMultiFactor(), (double)0.01);
            Assert.assertEquals((double)0.2f, (double)bucketCache.getMemoryFactor(), (double)0.01);
            Assert.assertEquals((long)100L, (long)bucketCache.getQueueAdditionWaitTime());
            Assert.assertEquals((long)500L, (long)bucketCache.getBucketcachePersistInterval());
            Assert.assertEquals((long)1000L, (long)bucketCache.getPersistenceChunkSize());
        }
        finally {
            if (bucketCache != null) {
                bucketCache.shutdown();
            }
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    @Test
    public void testNotifyFileCachingCompletedSuccess() throws Exception {
        BucketCache bucketCache = null;
        try {
            Path filePath = new Path(HBASE_TESTING_UTILITY.getDataTestDir(), "testNotifyFileCachingCompletedSuccess");
            bucketCache = this.testNotifyFileCachingCompletedForTenBlocks(filePath, 10, false);
            if (bucketCache.getStats().getFailedInserts() > 0L) {
                LOG.info("There were {} fail inserts, will assert if total blocks in backingMap equals (10 - failInserts) and file isn't listed as fully cached.", (Object)bucketCache.getStats().getFailedInserts());
                Assert.assertEquals((long)(10L - bucketCache.getStats().getFailedInserts()), (long)bucketCache.backingMap.size());
                Assert.assertFalse((boolean)bucketCache.fullyCachedFiles.containsKey(filePath.getName()));
            } else {
                Assert.assertTrue((boolean)bucketCache.fullyCachedFiles.containsKey(filePath.getName()));
            }
        }
        finally {
            if (bucketCache != null) {
                bucketCache.shutdown();
            }
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    @Test
    public void testNotifyFileCachingCompletedForEncodedDataSuccess() throws Exception {
        BucketCache bucketCache = null;
        try {
            Path filePath = new Path(HBASE_TESTING_UTILITY.getDataTestDir(), "testNotifyFileCachingCompletedForEncodedDataSuccess");
            bucketCache = this.testNotifyFileCachingCompletedForTenBlocks(filePath, 10, true);
            if (bucketCache.getStats().getFailedInserts() > 0L) {
                LOG.info("There were {} fail inserts, will assert if total blocks in backingMap equals (10 - failInserts) and file isn't listed as fully cached.", (Object)bucketCache.getStats().getFailedInserts());
                Assert.assertEquals((long)(10L - bucketCache.getStats().getFailedInserts()), (long)bucketCache.backingMap.size());
                Assert.assertFalse((boolean)bucketCache.fullyCachedFiles.containsKey(filePath.getName()));
            } else {
                Assert.assertTrue((boolean)bucketCache.fullyCachedFiles.containsKey(filePath.getName()));
            }
        }
        finally {
            if (bucketCache != null) {
                bucketCache.shutdown();
            }
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    @Test
    public void testNotifyFileCachingCompletedNotAllCached() throws Exception {
        BucketCache bucketCache = null;
        try {
            Path filePath = new Path(HBASE_TESTING_UTILITY.getDataTestDir(), "testNotifyFileCachingCompletedNotAllCached");
            bucketCache = this.testNotifyFileCachingCompletedForTenBlocks(filePath, 12, false);
            Assert.assertFalse((boolean)bucketCache.fullyCachedFiles.containsKey(filePath.getName()));
        }
        finally {
            if (bucketCache != null) {
                bucketCache.shutdown();
            }
            HBASE_TESTING_UTILITY.cleanupTestDir();
        }
    }

    private BucketCache testNotifyFileCachingCompletedForTenBlocks(Path filePath, int totalBlocksToCheck, boolean encoded) throws Exception {
        CacheTestUtils.HFileBlockPair[] hfileBlockPairs;
        Path dataTestDir = this.createAndGetTestDir();
        String ioEngineName = "file:" + dataTestDir + "/bucketNoRecycler.cache";
        BucketCache bucketCache = new BucketCache(ioEngineName, 0x2000000L, this.constructedBlockSize, this.constructedBlockSizes, 1, 1, null);
        long usedByteSize = bucketCache.getAllocator().getUsedSize();
        Assert.assertEquals((long)0L, (long)usedByteSize);
        for (CacheTestUtils.HFileBlockPair hfileBlockPair : hfileBlockPairs = CacheTestUtils.generateBlocksForPath(this.constructedBlockSize, 10, filePath, encoded)) {
            bucketCache.cacheBlock(hfileBlockPair.getBlockName(), (Cacheable)hfileBlockPair.getBlock(), false, true);
        }
        bucketCache.notifyFileCachingCompleted(filePath, totalBlocksToCheck, totalBlocksToCheck, (long)(totalBlocksToCheck * this.constructedBlockSize));
        return bucketCache;
    }

    @Test
    public void testEvictOrphansOutOfGracePeriod() throws Exception {
        BucketCache bucketCache = this.testEvictOrphans(0L);
        Assert.assertEquals((long)10L, (long)bucketCache.getBackingMap().size());
        Assert.assertEquals((long)0L, (long)bucketCache.blocksByHFile.stream().filter(key -> key.getHfileName().equals("testEvictOrphans-orphan")).count());
    }

    @Test
    public void testEvictOrphansWithinGracePeriod() throws Exception {
        BucketCache bucketCache = this.testEvictOrphans(3600000L);
        Assert.assertEquals((long)18L, (long)bucketCache.getBackingMap().size());
        Assert.assertTrue((bucketCache.blocksByHFile.stream().filter(key -> key.getHfileName().equals("testEvictOrphans-orphan")).count() > 0L ? 1 : 0) != 0);
    }

    private BucketCache testEvictOrphans(long orphanEvictionGracePeriod) throws Exception {
        Path validFile = new Path(HBASE_TESTING_UTILITY.getDataTestDir(), "testEvictOrphans-valid");
        Path orphanFile = new Path(HBASE_TESTING_UTILITY.getDataTestDir(), "testEvictOrphans-orphan");
        HashMap<String, HRegion> onlineRegions = new HashMap<String, HRegion>();
        ArrayList<HStore> stores = new ArrayList<HStore>();
        ArrayList<HStoreFile> storeFiles = new ArrayList<HStoreFile>();
        HRegion mockedRegion = (HRegion)Mockito.mock(HRegion.class);
        HStore mockedStore = (HStore)Mockito.mock(HStore.class);
        HStoreFile mockedStoreFile = (HStoreFile)Mockito.mock(HStoreFile.class);
        Mockito.when((Object)mockedStoreFile.getPath()).thenReturn((Object)validFile);
        storeFiles.add(mockedStoreFile);
        Mockito.when((Object)mockedStore.getStorefiles()).thenReturn(storeFiles);
        stores.add(mockedStore);
        Mockito.when((Object)mockedRegion.getStores()).thenReturn(stores);
        onlineRegions.put("mocked_region", mockedRegion);
        HBASE_TESTING_UTILITY.getConfiguration().setDouble("hbase.bucketcache.minfactor", 0.99);
        HBASE_TESTING_UTILITY.getConfiguration().setDouble("hbase.bucketcache.acceptfactor", 1.0);
        HBASE_TESTING_UTILITY.getConfiguration().setDouble("hbase.bucketcache.extrafreefactor", 0.01);
        HBASE_TESTING_UTILITY.getConfiguration().setLong("hbase.bucketcache.block.orphan.evictgraceperiod.seconds", orphanEvictionGracePeriod);
        BucketCache bucketCache = new BucketCache(this.ioEngineName, (long)((this.constructedBlockSize + 1024) * 21), this.constructedBlockSize, new int[]{this.constructedBlockSize + 1024}, 1, 1, null, 60000, HBASE_TESTING_UTILITY.getConfiguration(), onlineRegions);
        CacheTestUtils.HFileBlockPair[] validBlockPairs = CacheTestUtils.generateBlocksForPath(this.constructedBlockSize, 10, validFile);
        CacheTestUtils.HFileBlockPair[] orphanBlockPairs = CacheTestUtils.generateBlocksForPath(this.constructedBlockSize, 10, orphanFile);
        for (CacheTestUtils.HFileBlockPair pair : validBlockPairs) {
            bucketCache.cacheBlockWithWait(pair.getBlockName(), (Cacheable)pair.getBlock(), false, true);
        }
        TestBucketCache.waitUntilAllFlushedToBucket(bucketCache);
        Assert.assertEquals((long)10L, (long)bucketCache.getBackingMap().size());
        bucketCache.freeSpace("test");
        Assert.assertEquals((long)10L, (long)bucketCache.getBackingMap().size());
        for (CacheTestUtils.HFileBlockPair pair : orphanBlockPairs) {
            bucketCache.cacheBlockWithWait(pair.getBlockName(), (Cacheable)pair.getBlock(), false, true);
        }
        TestBucketCache.waitUntilAllFlushedToBucket(bucketCache);
        Assert.assertEquals((long)20L, (long)bucketCache.getBackingMap().size());
        bucketCache.freeSpace("test");
        return bucketCache;
    }

    private static class MockedBucketCache
    extends BucketCache {
        public MockedBucketCache(String ioEngineName, long capacity, int blockSize, int[] bucketSizes, int writerThreads, int writerQLen, String persistencePath) throws IOException {
            super(ioEngineName, capacity, blockSize, bucketSizes, writerThreads, writerQLen, persistencePath);
        }

        public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf, boolean inMemory) {
            super.cacheBlock(cacheKey, buf, inMemory);
        }

        public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf) {
            super.cacheBlock(cacheKey, buf);
        }
    }
}

