/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.Closeable;
import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.key.kms.server.MiniKMS;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.TestSecureEncryptionZoneWithKMS;
import org.apache.hadoop.hdfs.client.CreateEncryptionZoneFlag;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.http.HttpConfig;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.minikdc.MiniKdc;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.ssl.KeyStoreTestUtil;
import org.apache.hadoop.util.Lists;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.runners.MethodSorters;

@FixMethodOrder(value=MethodSorters.NAME_ASCENDING)
public class TestTrashWithSecureEncryptionZones {
    private static HdfsConfiguration baseConf;
    private static File baseDir;
    private static final EnumSet<CreateEncryptionZoneFlag> PROVISION_TRASH;
    private static final String HDFS_USER_NAME = "hdfs";
    private static final String SPNEGO_USER_NAME = "HTTP";
    private static final String OOZIE_USER_NAME = "oozie";
    private static final String OOZIE_PROXIED_USER_NAME = "oozie_user";
    private static String hdfsPrincipal;
    private static String spnegoPrincipal;
    private static String keytab;
    private static MiniKdc kdc;
    private static MiniKMS miniKMS;
    private static final String TEST_KEY = "test_key";
    private static final Path CURRENT;
    private static MiniDFSCluster cluster;
    private static HdfsConfiguration conf;
    private static FileSystem fs;
    private static HdfsAdmin dfsAdmin;
    private static Configuration clientConf;
    private static FsShell shell;
    private static AtomicInteger zoneCounter;
    private static AtomicInteger fileCounter;
    private static final int LEN = 8192;

    public static File getTestDir() throws Exception {
        File file = new File("dummy");
        file = file.getAbsoluteFile();
        file = file.getParentFile();
        file = new File(file, "target");
        if (!(file = new File(file, UUID.randomUUID().toString())).mkdirs()) {
            throw new RuntimeException("Could not create test directory: " + file);
        }
        return file;
    }

    @BeforeClass
    public static void init() throws Exception {
        baseDir = TestTrashWithSecureEncryptionZones.getTestDir();
        FileUtil.fullyDelete((File)baseDir);
        Assert.assertTrue((boolean)baseDir.mkdirs());
        Properties kdcConf = MiniKdc.createConf();
        kdc = new MiniKdc(kdcConf, baseDir);
        kdc.start();
        baseConf = new HdfsConfiguration();
        SecurityUtil.setAuthenticationMethod((UserGroupInformation.AuthenticationMethod)UserGroupInformation.AuthenticationMethod.KERBEROS, (Configuration)baseConf);
        UserGroupInformation.setConfiguration((Configuration)baseConf);
        Assert.assertTrue((String)"Expected configuration to enable security", (boolean)UserGroupInformation.isSecurityEnabled());
        File keytabFile = new File(baseDir, "test.keytab");
        keytab = keytabFile.getAbsolutePath();
        String krbInstance = Path.WINDOWS ? "127.0.0.1" : "localhost";
        kdc.createPrincipal(keytabFile, new String[]{"hdfs/" + krbInstance, "HTTP/" + krbInstance, "oozie/" + krbInstance, "oozie_user/" + krbInstance});
        hdfsPrincipal = "hdfs/" + krbInstance + "@" + kdc.getRealm();
        spnegoPrincipal = "HTTP/" + krbInstance + "@" + kdc.getRealm();
        baseConf.set("dfs.namenode.kerberos.principal", hdfsPrincipal);
        baseConf.set("dfs.namenode.keytab.file", keytab);
        baseConf.set("dfs.datanode.kerberos.principal", hdfsPrincipal);
        baseConf.set("dfs.datanode.keytab.file", keytab);
        baseConf.set("dfs.web.authentication.kerberos.principal", spnegoPrincipal);
        baseConf.setBoolean("dfs.block.access.token.enable", true);
        baseConf.set("dfs.data.transfer.protection", "authentication");
        baseConf.set("dfs.http.policy", HttpConfig.Policy.HTTPS_ONLY.name());
        baseConf.set("dfs.namenode.https-address", "localhost:0");
        baseConf.set("dfs.datanode.https.address", "localhost:0");
        baseConf.set("dfs.journalnode.https-address", "localhost:0");
        baseConf.setInt("ipc.client.connect.max.retries.on.sasl", 10);
        baseConf.set("hadoop.security.kms.client.encrypted.key.cache.size", "4");
        baseConf.set("hadoop.security.kms.client.encrypted.key.cache.low-watermark", "0.5");
        String keystoresDir = baseDir.getAbsolutePath();
        String sslConfDir = KeyStoreTestUtil.getClasspathDir(TestSecureEncryptionZoneWithKMS.class);
        KeyStoreTestUtil.setupSSLConfig((String)keystoresDir, (String)sslConfDir, (Configuration)baseConf, (boolean)false);
        baseConf.set("dfs.client.https.keystore.resource", KeyStoreTestUtil.getClientSSLConfigFileName());
        baseConf.set("dfs.https.server.keystore.resource", KeyStoreTestUtil.getServerSSLConfigFileName());
        File kmsFile = new File(baseDir, "kms-site.xml");
        if (kmsFile.exists()) {
            FileUtil.fullyDelete((File)kmsFile);
        }
        Configuration kmsConf = new Configuration(true);
        kmsConf.set("hadoop.kms.key.provider.uri", "jceks://file@" + new Path(baseDir.toString(), "kms.keystore").toUri());
        kmsConf.set("hadoop.kms.authentication.type", "kerberos");
        kmsConf.set("hadoop.kms.authentication.kerberos.keytab", keytab);
        kmsConf.set("hadoop.kms.authentication.kerberos.principal", "HTTP/localhost");
        kmsConf.set("hadoop.kms.authentication.kerberos.name.rules", "DEFAULT");
        kmsConf.set("hadoop.kms.acl.GENERATE_EEK", HDFS_USER_NAME);
        FileWriter writer = new FileWriter(kmsFile);
        kmsConf.writeXml((Writer)writer);
        ((Writer)writer).close();
        MiniKMS.Builder miniKMSBuilder = new MiniKMS.Builder();
        miniKMS = miniKMSBuilder.setKmsConfDir(baseDir).build();
        miniKMS.start();
        baseConf.set("hadoop.security.key.provider.path", TestTrashWithSecureEncryptionZones.getKeyProviderURI());
        baseConf.setBoolean("dfs.namenode.delegation.token.always-use", true);
        conf = new HdfsConfiguration((Configuration)baseConf);
        cluster = new MiniDFSCluster.Builder((Configuration)conf).build();
        cluster.waitActive();
        fs = cluster.getFileSystem();
        dfsAdmin = new HdfsAdmin(cluster.getURI(), (Configuration)conf);
        cluster.waitActive();
        DFSTestUtil.createKey(TEST_KEY, cluster, (Configuration)conf);
        clientConf = new Configuration((Configuration)conf);
        clientConf.setLong("fs.trash.interval", 1L);
        shell = new FsShell(clientConf);
    }

    @AfterClass
    public static void destroy() {
        IOUtils.cleanupWithLogger(null, (Closeable[])new Closeable[]{fs});
        if (cluster != null) {
            cluster.shutdown();
            cluster = null;
        }
        if (kdc != null) {
            kdc.stop();
        }
        if (miniKMS != null) {
            miniKMS.stop();
        }
        FileUtil.fullyDelete((File)baseDir);
    }

    private static String getKeyProviderURI() {
        return "kms://" + miniKMS.getKMSUrl().toExternalForm().replace("://", "@");
    }

    @Test
    public void testTrashCheckpoint() throws Exception {
        Path zone1 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone1);
        Path zone2 = new Path(zone1 + "/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone2);
        dfsAdmin.createEncryptionZone(zone2, TEST_KEY, PROVISION_TRASH);
        Path encFile1 = new Path(zone2, "encFile" + fileCounter.getAndIncrement());
        DFSTestUtil.createFile(fs, encFile1, 8192L, (short)1, 65261L);
        Path trashDir = new Path(zone2, ".Trash" + "/" + HDFS_USER_NAME + "/" + CURRENT);
        String trashPath = trashDir.toString() + encFile1.toString();
        Path deletedFile = this.verifyTrashLocationWithShellDelete(encFile1);
        Assert.assertEquals((String)("Deleted file not at the expected trash location: " + trashPath), (Object)trashPath, (Object)deletedFile.toUri().getPath());
        trashPath = fs.getHomeDirectory().toUri().getPath() + "/" + ".Trash" + "/" + CURRENT + zone2;
        Path deletedDir = this.verifyTrashLocationWithShellDelete(zone2);
        Assert.assertEquals((String)("Deleted zone not at the expected trash location: " + trashPath), (Object)trashPath, (Object)deletedDir.toUri().getPath());
    }

    @Test
    public void testTrashExpunge() throws Exception {
        Path zone1 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone1);
        Path zone2 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone2);
        dfsAdmin.createEncryptionZone(zone1, TEST_KEY, PROVISION_TRASH);
        Path file1 = new Path(zone1, "encFile" + fileCounter.getAndIncrement());
        Path file2 = new Path(zone2, "file" + fileCounter.getAndIncrement());
        DFSTestUtil.createFile(fs, file1, 8192L, (short)1, 65261L);
        DFSTestUtil.createFile(fs, file2, 8192L, (short)1, 65261L);
        ArrayList trashPaths = Lists.newArrayList();
        trashPaths.add(this.verifyTrashLocationWithShellDelete(file1));
        trashPaths.add(this.verifyTrashLocationWithShellDelete(file2));
        this.verifyTrashExpunge(trashPaths);
        Path file3 = new Path(zone1, "encFile" + fileCounter.getAndIncrement());
        DFSTestUtil.createFile(fs, file3, 8192L, (short)1, 65261L);
        Path trashPath = this.verifyTrashLocationWithShellDelete(file3);
        DFSTestUtil.verifyDelete(shell, fs, zone1, true);
        this.verifyTrashExpunge(Lists.newArrayList((Object[])new Path[]{trashPath}));
    }

    @Test
    public void testDeleteWithSkipTrash() throws Exception {
        Path zone1 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone1);
        Path encFile1 = new Path(zone1, "encFile" + fileCounter.getAndIncrement());
        Path encFile2 = new Path(zone1, "encFile" + fileCounter.getAndIncrement());
        DFSTestUtil.createFile(fs, encFile1, 8192L, (short)1, 65261L);
        DFSTestUtil.createFile(fs, encFile2, 8192L, (short)1, 65261L);
        this.verifyDeleteWithSkipTrash(encFile1);
        DFSTestUtil.verifyDelete(shell, fs, encFile2, true);
    }

    @Test
    public void testDeleteEmptyDirectory() throws Exception {
        Path zone1 = new Path("/zone" + zoneCounter.getAndIncrement());
        Path zone2 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone1);
        fs.mkdirs(zone2);
        Path trashDir1 = new Path(shell.getCurrentTrashDir(zone1) + "/" + zone1);
        Path trashDir2 = new Path(shell.getCurrentTrashDir(zone1) + "/" + zone2);
        String[] argv1 = new String[]{"-rm", "-r", zone1.toString()};
        int res = ToolRunner.run((Tool)shell, (String[])argv1);
        Assert.assertEquals((String)"rm failed", (long)0L, (long)res);
        Assert.assertTrue((String)("Empty directory not deleted even with -r : " + trashDir1), (boolean)fs.exists(trashDir1));
        String[] argv2 = new String[]{"-rm", zone2.toString()};
        res = ToolRunner.run((Tool)shell, (String[])argv2);
        Assert.assertEquals((String)"rm on empty directory did not fail", (long)1L, (long)res);
        Assert.assertTrue((String)("Empty directory deleted without -r : " + trashDir2), (!fs.exists(trashDir2) ? 1 : 0) != 0);
    }

    @Test
    public void testDeleteFromTrashWithinEZ() throws Exception {
        Path zone1 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone1);
        dfsAdmin.createEncryptionZone(zone1, TEST_KEY, PROVISION_TRASH);
        Path encFile1 = new Path(zone1, "encFile" + fileCounter.getAndIncrement());
        DFSTestUtil.createFile(fs, encFile1, 8192L, (short)1, 65261L);
        Path trashFile = new Path(shell.getCurrentTrashDir(encFile1) + "/" + encFile1);
        String[] argv = new String[]{"-rm", "-r", encFile1.toString()};
        int res = ToolRunner.run((Tool)shell, (String[])argv);
        Assert.assertEquals((String)"rm failed", (long)0L, (long)res);
        String[] argvDeleteTrash = new String[]{"-rm", "-r", trashFile.toString()};
        int resDeleteTrash = ToolRunner.run((Tool)shell, (String[])argvDeleteTrash);
        Assert.assertEquals((String)"rm failed", (long)0L, (long)resDeleteTrash);
        Assert.assertFalse((String)("File deleted from Trash : " + trashFile), (boolean)fs.exists(trashFile));
    }

    @Test
    public void testTrashRetentionAfterNamenodeRestart() throws Exception {
        Path zone1 = new Path("/zone" + zoneCounter.getAndIncrement());
        fs.mkdirs(zone1);
        dfsAdmin.createEncryptionZone(zone1, TEST_KEY, PROVISION_TRASH);
        Path encFile1 = new Path(zone1, "encFile" + fileCounter.getAndIncrement());
        DFSTestUtil.createFile(fs, encFile1, 8192L, (short)1, 65261L);
        Path trashFile = new Path(shell.getCurrentTrashDir(encFile1) + "/" + encFile1);
        String[] argv = new String[]{"-rm", "-r", encFile1.toString()};
        int res = ToolRunner.run((Tool)shell, (String[])argv);
        Assert.assertEquals((String)"rm failed", (long)0L, (long)res);
        Assert.assertTrue((String)("File not in trash : " + trashFile), (boolean)fs.exists(trashFile));
        cluster.restartNameNode(0);
        cluster.waitActive();
        fs = cluster.getFileSystem();
        Assert.assertTrue((String)("On Namenode restart, file deleted from trash : " + trashFile), (boolean)fs.exists(trashFile));
    }

    private Path verifyTrashLocationWithShellDelete(Path path) throws Exception {
        Path trashFile = new Path(shell.getCurrentTrashDir(path) + "/" + path);
        File deletedFile = new File(String.valueOf(trashFile));
        Assert.assertFalse((String)"File already present in Trash before delete", (boolean)deletedFile.exists());
        DFSTestUtil.verifyDelete(shell, fs, path, trashFile, true);
        return trashFile;
    }

    private void verifyTrashExpunge(List<Path> trashFiles) throws Exception {
        String[] argv = new String[]{"-expunge"};
        int res = ToolRunner.run((Tool)shell, (String[])argv);
        Assert.assertEquals((String)"expunge failed", (long)0L, (long)res);
        for (Path trashFile : trashFiles) {
            Assert.assertFalse((String)("File exists in trash after expunge : " + trashFile), (boolean)fs.exists(trashFile));
        }
    }

    private void verifyDeleteWithSkipTrash(Path path) throws Exception {
        Assert.assertTrue((String)(path + " file does not exist"), (boolean)fs.exists(path));
        Path trashFile = new Path(shell.getCurrentTrashDir(path) + "/" + path);
        String[] argv = new String[]{"-rm", "-r", "-skipTrash", path.toString()};
        int res = ToolRunner.run((Tool)shell, (String[])argv);
        Assert.assertEquals((String)"rm failed", (long)0L, (long)res);
        Assert.assertFalse((String)"File in trash even with -skipTrash", (boolean)fs.exists(trashFile));
    }

    static {
        PROVISION_TRASH = EnumSet.of(CreateEncryptionZoneFlag.PROVISION_TRASH);
        CURRENT = new Path("Current");
        zoneCounter = new AtomicInteger(1);
        fileCounter = new AtomicInteger(1);
    }
}

