# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class TestFileMappingData
        # @param [Hash<String, Array<String>>] test_to_sources Test files
        #   mapped to all source files they cover
        # @param [Hash<String, Array<String>>] tests_to_categories Test files
        #   mapped to their feature categories
        # @param [Hash<String, Hash>] feature_categories_to_teams Feature categories
        #   mapped to their org hierarchy (group, stage, section)
        def initialize(test_to_sources, tests_to_categories: {}, feature_categories_to_teams: {})
          @test_to_sources = test_to_sources
          @tests_to_categories = tests_to_categories
          @feature_categories_to_teams = feature_categories_to_teams
        end

        # @return [Array<Hash<Symbol, String>>] Mapping data formatted for database insertion
        # @example Return value
        #   [
        #     { test_file: "spec/models/user_spec.rb", source_file: "app/models/user.rb",
        #       category: "team_planning", group: "project_management", stage: "plan", section: "dev" },
        #     ...
        #   ]
        def as_db_table
          @test_to_sources.flat_map do |test_file, source_files|
            category = @tests_to_categories[test_file]&.first || ''
            team = @feature_categories_to_teams[category] || {}

            source_files.map do |source_file|
              {
                test_file: test_file,
                source_file: source_file,
                category: category,
                group: team[:group] || '',
                stage: team[:stage] || '',
                section: team[:section] || ''
              }
            end
          end
        end
      end
    end
  end
end
