﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pipes/Pipes_EXPORTS.h>
#include <aws/pipes/model/EcsEnvironmentFile.h>
#include <aws/pipes/model/EcsEnvironmentVariable.h>
#include <aws/pipes/model/EcsResourceRequirement.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Pipes {
namespace Model {

/**
 * <p>The overrides that are sent to a container. An empty container override can
 * be passed in. An example of an empty container override is
 * <code>{"containerOverrides": [ ] }</code>. If a non-empty container override is
 * specified, the <code>name</code> parameter must be included.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/EcsContainerOverride">AWS
 * API Reference</a></p>
 */
class EcsContainerOverride {
 public:
  AWS_PIPES_API EcsContainerOverride() = default;
  AWS_PIPES_API EcsContainerOverride(Aws::Utils::Json::JsonView jsonValue);
  AWS_PIPES_API EcsContainerOverride& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PIPES_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The command to send to the container that overrides the default command from
   * the Docker image or the task definition. You must also specify a container
   * name.</p>
   */
  inline const Aws::Vector<Aws::String>& GetCommand() const { return m_command; }
  inline bool CommandHasBeenSet() const { return m_commandHasBeenSet; }
  template <typename CommandT = Aws::Vector<Aws::String>>
  void SetCommand(CommandT&& value) {
    m_commandHasBeenSet = true;
    m_command = std::forward<CommandT>(value);
  }
  template <typename CommandT = Aws::Vector<Aws::String>>
  EcsContainerOverride& WithCommand(CommandT&& value) {
    SetCommand(std::forward<CommandT>(value));
    return *this;
  }
  template <typename CommandT = Aws::String>
  EcsContainerOverride& AddCommand(CommandT&& value) {
    m_commandHasBeenSet = true;
    m_command.emplace_back(std::forward<CommandT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of <code>cpu</code> units reserved for the container, instead of
   * the default value from the task definition. You must also specify a container
   * name.</p>
   */
  inline int GetCpu() const { return m_cpu; }
  inline bool CpuHasBeenSet() const { return m_cpuHasBeenSet; }
  inline void SetCpu(int value) {
    m_cpuHasBeenSet = true;
    m_cpu = value;
  }
  inline EcsContainerOverride& WithCpu(int value) {
    SetCpu(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The environment variables to send to the container. You can add new
   * environment variables, which are added to the container at launch, or you can
   * override the existing environment variables from the Docker image or the task
   * definition. You must also specify a container name.</p>
   */
  inline const Aws::Vector<EcsEnvironmentVariable>& GetEnvironment() const { return m_environment; }
  inline bool EnvironmentHasBeenSet() const { return m_environmentHasBeenSet; }
  template <typename EnvironmentT = Aws::Vector<EcsEnvironmentVariable>>
  void SetEnvironment(EnvironmentT&& value) {
    m_environmentHasBeenSet = true;
    m_environment = std::forward<EnvironmentT>(value);
  }
  template <typename EnvironmentT = Aws::Vector<EcsEnvironmentVariable>>
  EcsContainerOverride& WithEnvironment(EnvironmentT&& value) {
    SetEnvironment(std::forward<EnvironmentT>(value));
    return *this;
  }
  template <typename EnvironmentT = EcsEnvironmentVariable>
  EcsContainerOverride& AddEnvironment(EnvironmentT&& value) {
    m_environmentHasBeenSet = true;
    m_environment.emplace_back(std::forward<EnvironmentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of files containing the environment variables to pass to a container,
   * instead of the value from the container definition.</p>
   */
  inline const Aws::Vector<EcsEnvironmentFile>& GetEnvironmentFiles() const { return m_environmentFiles; }
  inline bool EnvironmentFilesHasBeenSet() const { return m_environmentFilesHasBeenSet; }
  template <typename EnvironmentFilesT = Aws::Vector<EcsEnvironmentFile>>
  void SetEnvironmentFiles(EnvironmentFilesT&& value) {
    m_environmentFilesHasBeenSet = true;
    m_environmentFiles = std::forward<EnvironmentFilesT>(value);
  }
  template <typename EnvironmentFilesT = Aws::Vector<EcsEnvironmentFile>>
  EcsContainerOverride& WithEnvironmentFiles(EnvironmentFilesT&& value) {
    SetEnvironmentFiles(std::forward<EnvironmentFilesT>(value));
    return *this;
  }
  template <typename EnvironmentFilesT = EcsEnvironmentFile>
  EcsContainerOverride& AddEnvironmentFiles(EnvironmentFilesT&& value) {
    m_environmentFilesHasBeenSet = true;
    m_environmentFiles.emplace_back(std::forward<EnvironmentFilesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The hard limit (in MiB) of memory to present to the container, instead of the
   * default value from the task definition. If your container attempts to exceed the
   * memory specified here, the container is killed. You must also specify a
   * container name.</p>
   */
  inline int GetMemory() const { return m_memory; }
  inline bool MemoryHasBeenSet() const { return m_memoryHasBeenSet; }
  inline void SetMemory(int value) {
    m_memoryHasBeenSet = true;
    m_memory = value;
  }
  inline EcsContainerOverride& WithMemory(int value) {
    SetMemory(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The soft limit (in MiB) of memory to reserve for the container, instead of
   * the default value from the task definition. You must also specify a container
   * name.</p>
   */
  inline int GetMemoryReservation() const { return m_memoryReservation; }
  inline bool MemoryReservationHasBeenSet() const { return m_memoryReservationHasBeenSet; }
  inline void SetMemoryReservation(int value) {
    m_memoryReservationHasBeenSet = true;
    m_memoryReservation = value;
  }
  inline EcsContainerOverride& WithMemoryReservation(int value) {
    SetMemoryReservation(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the container that receives the override. This parameter is
   * required if any override is specified.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  EcsContainerOverride& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type and amount of a resource to assign to a container, instead of the
   * default value from the task definition. The only supported resource is a
   * GPU.</p>
   */
  inline const Aws::Vector<EcsResourceRequirement>& GetResourceRequirements() const { return m_resourceRequirements; }
  inline bool ResourceRequirementsHasBeenSet() const { return m_resourceRequirementsHasBeenSet; }
  template <typename ResourceRequirementsT = Aws::Vector<EcsResourceRequirement>>
  void SetResourceRequirements(ResourceRequirementsT&& value) {
    m_resourceRequirementsHasBeenSet = true;
    m_resourceRequirements = std::forward<ResourceRequirementsT>(value);
  }
  template <typename ResourceRequirementsT = Aws::Vector<EcsResourceRequirement>>
  EcsContainerOverride& WithResourceRequirements(ResourceRequirementsT&& value) {
    SetResourceRequirements(std::forward<ResourceRequirementsT>(value));
    return *this;
  }
  template <typename ResourceRequirementsT = EcsResourceRequirement>
  EcsContainerOverride& AddResourceRequirements(ResourceRequirementsT&& value) {
    m_resourceRequirementsHasBeenSet = true;
    m_resourceRequirements.emplace_back(std::forward<ResourceRequirementsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_command;

  int m_cpu{0};

  Aws::Vector<EcsEnvironmentVariable> m_environment;

  Aws::Vector<EcsEnvironmentFile> m_environmentFiles;

  int m_memory{0};

  int m_memoryReservation{0};

  Aws::String m_name;

  Aws::Vector<EcsResourceRequirement> m_resourceRequirements;
  bool m_commandHasBeenSet = false;
  bool m_cpuHasBeenSet = false;
  bool m_environmentHasBeenSet = false;
  bool m_environmentFilesHasBeenSet = false;
  bool m_memoryHasBeenSet = false;
  bool m_memoryReservationHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_resourceRequirementsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pipes
}  // namespace Aws
