﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lightsail/Lightsail_EXPORTS.h>
#include <aws/lightsail/model/ResourceLocation.h>
#include <aws/lightsail/model/ResourceType.h>
#include <aws/lightsail/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Lightsail {
namespace Model {

/**
 * <p>Describes an SSH key pair.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/lightsail-2016-11-28/KeyPair">AWS
 * API Reference</a></p>
 */
class KeyPair {
 public:
  AWS_LIGHTSAIL_API KeyPair() = default;
  AWS_LIGHTSAIL_API KeyPair(Aws::Utils::Json::JsonView jsonValue);
  AWS_LIGHTSAIL_API KeyPair& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_LIGHTSAIL_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The friendly name of the SSH key pair.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  KeyPair& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the key pair
   * (<code>arn:aws:lightsail:us-east-2:123456789101:KeyPair/05859e3d-331d-48ba-9034-12345EXAMPLE</code>).</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  KeyPair& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The support code. Include this code in your email to support when you have
   * questions about an instance or another resource in Lightsail. This code enables
   * our support team to look up your Lightsail information more easily.</p>
   */
  inline const Aws::String& GetSupportCode() const { return m_supportCode; }
  inline bool SupportCodeHasBeenSet() const { return m_supportCodeHasBeenSet; }
  template <typename SupportCodeT = Aws::String>
  void SetSupportCode(SupportCodeT&& value) {
    m_supportCodeHasBeenSet = true;
    m_supportCode = std::forward<SupportCodeT>(value);
  }
  template <typename SupportCodeT = Aws::String>
  KeyPair& WithSupportCode(SupportCodeT&& value) {
    SetSupportCode(std::forward<SupportCodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp when the key pair was created
   * (<code>1479816991.349</code>).</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  KeyPair& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The region name and Availability Zone where the key pair was created.</p>
   */
  inline const ResourceLocation& GetLocation() const { return m_location; }
  inline bool LocationHasBeenSet() const { return m_locationHasBeenSet; }
  template <typename LocationT = ResourceLocation>
  void SetLocation(LocationT&& value) {
    m_locationHasBeenSet = true;
    m_location = std::forward<LocationT>(value);
  }
  template <typename LocationT = ResourceLocation>
  KeyPair& WithLocation(LocationT&& value) {
    SetLocation(std::forward<LocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource type (usually <code>KeyPair</code>).</p>
   */
  inline ResourceType GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  inline void SetResourceType(ResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline KeyPair& WithResourceType(ResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tag keys and optional values for the resource. For more information about
   * tags in Lightsail, see the <a
   * href="https://docs.aws.amazon.com/lightsail/latest/userguide/amazon-lightsail-tags">Amazon
   * Lightsail Developer Guide</a>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  KeyPair& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  KeyPair& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The RSA fingerprint of the key pair.</p>
   */
  inline const Aws::String& GetFingerprint() const { return m_fingerprint; }
  inline bool FingerprintHasBeenSet() const { return m_fingerprintHasBeenSet; }
  template <typename FingerprintT = Aws::String>
  void SetFingerprint(FingerprintT&& value) {
    m_fingerprintHasBeenSet = true;
    m_fingerprint = std::forward<FingerprintT>(value);
  }
  template <typename FingerprintT = Aws::String>
  KeyPair& WithFingerprint(FingerprintT&& value) {
    SetFingerprint(std::forward<FingerprintT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_arn;

  Aws::String m_supportCode;

  Aws::Utils::DateTime m_createdAt{};

  ResourceLocation m_location;

  ResourceType m_resourceType{ResourceType::NOT_SET};

  Aws::Vector<Tag> m_tags;

  Aws::String m_fingerprint;
  bool m_nameHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_supportCodeHasBeenSet = false;
  bool m_createdAtHasBeenSet = false;
  bool m_locationHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_fingerprintHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lightsail
}  // namespace Aws
