/*
 * Decompiled with CFR 0.152.
 */
package org.apache.polaris.extension.auth.opa.token;

import com.auth0.jwt.JWT;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.common.base.Preconditions;
import jakarta.annotation.Nullable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.polaris.extension.auth.opa.token.BearerTokenProvider;
import org.apache.polaris.nosql.async.AsyncExec;
import org.apache.polaris.nosql.async.Cancelable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBearerTokenProvider
implements BearerTokenProvider {
    private static final Logger logger = LoggerFactory.getLogger(FileBearerTokenProvider.class);
    private final Path tokenFilePath;
    private final Duration refreshInterval;
    private final boolean jwtExpirationRefresh;
    private final Duration jwtExpirationBuffer;
    private final Supplier<Instant> clock;
    private final AtomicBoolean refreshLock = new AtomicBoolean();
    private final AsyncExec asyncExec;
    private final CompletableFuture<String> initialTokenFuture = new CompletableFuture();
    private final long initialTokenWaitMillis;
    private volatile String cachedToken;
    private volatile Instant lastRefresh;
    private volatile Instant nextRefresh;
    private volatile Cancelable<?> refreshTask;

    public FileBearerTokenProvider(Path tokenFilePath, Duration refreshInterval, boolean jwtExpirationRefresh, Duration jwtExpirationBuffer, Duration initialTokenWait, AsyncExec asyncExec, Supplier<Instant> clock) {
        this.tokenFilePath = tokenFilePath;
        this.refreshInterval = refreshInterval;
        this.jwtExpirationRefresh = jwtExpirationRefresh;
        this.jwtExpirationBuffer = jwtExpirationBuffer;
        this.initialTokenWaitMillis = initialTokenWait.toMillis();
        this.clock = clock;
        this.asyncExec = asyncExec;
        Preconditions.checkState((boolean)Files.isReadable(tokenFilePath), (Object)"OPA token file does not exist or is not readable");
        this.nextRefresh = Instant.MIN;
        this.lastRefresh = Instant.MIN;
        this.scheduleRefreshAttempt(Duration.ZERO);
        logger.debug("Created file token provider for path: {} with refresh interval: {}, JWT expiration refresh: {}, JWT buffer: {}, next refresh: {}", new Object[]{tokenFilePath, refreshInterval, jwtExpirationRefresh, jwtExpirationBuffer, this.nextRefresh});
    }

    @Override
    public String getToken() {
        String token = this.cachedToken;
        if (token != null) {
            return token;
        }
        try {
            return this.initialTokenFuture.get(this.initialTokenWaitMillis, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to read initial OPA bearer token", e);
        }
    }

    @Override
    public void close() {
        this.cachedToken = null;
        Cancelable<?> task = this.refreshTask;
        if (task != null) {
            this.refreshTask.cancel();
        }
    }

    private void refreshTokenAttempt() {
        Duration delay;
        boolean isInitialRefresh;
        boolean bl = isInitialRefresh = this.cachedToken == null;
        if (this.doRefreshToken()) {
            delay = Duration.between(this.clock.get(), this.nextRefresh);
            if (isInitialRefresh) {
                this.initialTokenFuture.complete(this.cachedToken);
            }
        } else {
            delay = Duration.ofSeconds(1L);
        }
        this.scheduleRefreshAttempt(delay);
    }

    private void scheduleRefreshAttempt(Duration delay) {
        this.refreshTask = this.asyncExec.schedule(this::refreshTokenAttempt, delay);
    }

    private boolean doRefreshToken() {
        String newToken = this.loadTokenFromFile();
        if (newToken == null) {
            logger.debug("Couldn't load new bearer token from {}, will retry.", (Object)this.tokenFilePath);
            return false;
        }
        this.cachedToken = newToken;
        this.lastRefresh = this.clock.get();
        this.nextRefresh = this.calculateNextRefresh(this.cachedToken);
        logger.debug("Token refreshed from file: {} (token present: {}), next refresh: {}", new Object[]{this.tokenFilePath, this.cachedToken != null, this.nextRefresh});
        return true;
    }

    private Instant calculateNextRefresh(String token) {
        if (!this.jwtExpirationRefresh) {
            return this.lastRefresh.plus(this.refreshInterval);
        }
        Optional<Instant> expiration = this.getJwtExpirationTime(token);
        if (expiration.isPresent()) {
            Instant minRefreshTime;
            Instant refreshTime = expiration.get().minus(this.jwtExpirationBuffer);
            if (refreshTime.isBefore(minRefreshTime = this.clock.get().plus(Duration.ofSeconds(1L)))) {
                logger.warn("JWT expires too soon ({}), using minimum refresh interval instead", (Object)expiration.get());
                return this.lastRefresh.plus(this.refreshInterval);
            }
            logger.debug("Using JWT expiration-based refresh: token expires at {}, refreshing at {}", (Object)expiration.get(), (Object)refreshTime);
            return refreshTime;
        }
        logger.debug("Token is not a valid JWT or has no expiration, using fixed refresh interval");
        return this.lastRefresh.plus(this.refreshInterval);
    }

    @Nullable
    private String loadTokenFromFile() {
        try {
            String token = Files.readString(this.tokenFilePath, StandardCharsets.UTF_8).trim();
            if (!token.isEmpty()) {
                return token;
            }
        }
        catch (IOException e) {
            logger.debug("Failed to read token from file", (Throwable)e);
        }
        return null;
    }

    private Optional<Instant> getJwtExpirationTime(String token) {
        try {
            DecodedJWT decodedJWT = JWT.decode((String)token);
            Date expiresAt = decodedJWT.getExpiresAt();
            return expiresAt != null ? Optional.of(expiresAt.toInstant()) : Optional.empty();
        }
        catch (JWTDecodeException e) {
            logger.debug("Failed to decode JWT token: {}", (Object)e.getMessage());
            return Optional.empty();
        }
    }
}

