/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.transfer.model.CopyResult;
import java.util.Locale;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.s3a.RemoteFileChangedException;
import org.apache.hadoop.fs.s3a.S3ObjectAttributes;
import org.apache.hadoop.fs.store.LogExactlyOnce;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public abstract class ChangeDetectionPolicy {
    private static final Logger LOG = LoggerFactory.getLogger(ChangeDetectionPolicy.class);
    @VisibleForTesting
    public static final String CHANGE_DETECTED = "change detected on client";
    private final Mode mode;
    private final boolean requireVersion;
    private final LogExactlyOnce logNoVersionSupport = new LogExactlyOnce(LOG);

    protected ChangeDetectionPolicy(Mode mode, boolean requireVersion) {
        this.mode = mode;
        this.requireVersion = requireVersion;
    }

    public Mode getMode() {
        return this.mode;
    }

    public abstract Source getSource();

    public boolean isRequireVersion() {
        return this.requireVersion;
    }

    public LogExactlyOnce getLogNoVersionSupport() {
        return this.logNoVersionSupport;
    }

    public static ChangeDetectionPolicy getPolicy(Configuration configuration) {
        Mode mode = Mode.fromConfiguration(configuration);
        Source source = Source.fromConfiguration(configuration);
        boolean requireVersion = configuration.getBoolean("fs.s3a.change.detection.version.required", true);
        return ChangeDetectionPolicy.createPolicy(mode, source, requireVersion);
    }

    @VisibleForTesting
    public static ChangeDetectionPolicy createPolicy(Mode mode, Source source, boolean requireVersion) {
        switch (source) {
            case ETag: {
                return new ETagChangeDetectionPolicy(mode, requireVersion);
            }
            case VersionId: {
                return new VersionIdChangeDetectionPolicy(mode, requireVersion);
            }
        }
        return new NoChangeDetection();
    }

    public String toString() {
        return "Policy " + (Object)((Object)this.getSource()) + "/" + (Object)((Object)this.getMode());
    }

    public abstract String getRevisionId(ObjectMetadata var1, String var2);

    public abstract String getRevisionId(S3ObjectAttributes var1);

    public abstract String getRevisionId(CopyResult var1);

    public abstract void applyRevisionConstraint(GetObjectRequest var1, String var2);

    public abstract void applyRevisionConstraint(CopyObjectRequest var1, String var2);

    public abstract void applyRevisionConstraint(GetObjectMetadataRequest var1, String var2);

    public ImmutablePair<Boolean, RemoteFileChangedException> onChangeDetected(String revisionId, String newRevisionId, String uri, long position, String operation, long timesAlreadyDetected) {
        String positionText = position >= 0L ? " at " + position : "";
        switch (this.mode) {
            case None: {
                return new ImmutablePair((Object)false, null);
            }
            case Warn: {
                if (timesAlreadyDetected == 0L) {
                    LOG.warn(String.format("%s change detected on %s %s%s. Expected %s got %s", new Object[]{this.getSource(), operation, uri, positionText, revisionId, newRevisionId}));
                    return new ImmutablePair((Object)true, null);
                }
                return new ImmutablePair((Object)false, null);
            }
        }
        return new ImmutablePair((Object)true, (Object)new RemoteFileChangedException(uri, operation, String.format("%s change detected on client during %s%s. Expected %s got %s", new Object[]{this.getSource(), operation, positionText, revisionId, newRevisionId})));
    }

    static class NoChangeDetection
    extends ChangeDetectionPolicy {
        NoChangeDetection() {
            super(Mode.None, false);
        }

        @Override
        public Source getSource() {
            return Source.None;
        }

        @Override
        public String getRevisionId(ObjectMetadata objectMetadata, String uri) {
            return null;
        }

        @Override
        public String getRevisionId(S3ObjectAttributes s3ObjectAttributes) {
            return null;
        }

        @Override
        public String getRevisionId(CopyResult copyResult) {
            return null;
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest request, String revisionId) {
        }

        @Override
        public void applyRevisionConstraint(CopyObjectRequest request, String revisionId) {
        }

        @Override
        public void applyRevisionConstraint(GetObjectMetadataRequest request, String revisionId) {
        }

        @Override
        public String toString() {
            return "NoChangeDetection";
        }
    }

    static class VersionIdChangeDetectionPolicy
    extends ChangeDetectionPolicy {
        VersionIdChangeDetectionPolicy(Mode mode, boolean requireVersion) {
            super(mode, requireVersion);
        }

        @Override
        public String getRevisionId(ObjectMetadata objectMetadata, String uri) {
            String versionId = objectMetadata.getVersionId();
            if (versionId == null) {
                this.getLogNoVersionSupport().warn("fs.s3a.change.detection.mode set to " + (Object)((Object)Source.VersionId) + " but no versionId available while reading {}. Ensure your bucket has object versioning enabled. You may see inconsistent reads.", new Object[]{uri});
            }
            return versionId;
        }

        @Override
        public String getRevisionId(S3ObjectAttributes s3Attributes) {
            return s3Attributes.getVersionId();
        }

        @Override
        public String getRevisionId(CopyResult copyResult) {
            return copyResult.getVersionId();
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest request, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting get request to version {}", (Object)revisionId);
                request.withVersionId(revisionId);
            } else {
                LOG.debug("No version ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(CopyObjectRequest request, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting copy request to version {}", (Object)revisionId);
                request.withSourceVersionId(revisionId);
            } else {
                LOG.debug("No version ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(GetObjectMetadataRequest request, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting metadata request to version {}", (Object)revisionId);
                request.withVersionId(revisionId);
            } else {
                LOG.debug("No version ID to use as a constraint");
            }
        }

        @Override
        public Source getSource() {
            return Source.VersionId;
        }

        @Override
        public String toString() {
            return "VersionIdChangeDetectionPolicy mode=" + (Object)((Object)this.getMode());
        }
    }

    static class ETagChangeDetectionPolicy
    extends ChangeDetectionPolicy {
        ETagChangeDetectionPolicy(Mode mode, boolean requireVersion) {
            super(mode, requireVersion);
        }

        @Override
        public String getRevisionId(ObjectMetadata objectMetadata, String uri) {
            return objectMetadata.getETag();
        }

        @Override
        public String getRevisionId(S3ObjectAttributes s3Attributes) {
            return s3Attributes.getETag();
        }

        @Override
        public String getRevisionId(CopyResult copyResult) {
            return copyResult.getETag();
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest request, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting get request to etag {}", (Object)revisionId);
                request.withMatchingETagConstraint(revisionId);
            } else {
                LOG.debug("No etag revision ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(CopyObjectRequest request, String revisionId) {
            if (revisionId != null) {
                LOG.debug("Restricting copy request to etag {}", (Object)revisionId);
                request.withMatchingETagConstraint(revisionId);
            } else {
                LOG.debug("No etag revision ID to use as a constraint");
            }
        }

        @Override
        public void applyRevisionConstraint(GetObjectMetadataRequest request, String revisionId) {
            LOG.debug("Unable to restrict HEAD request to etag; will check later");
        }

        @Override
        public Source getSource() {
            return Source.ETag;
        }

        @Override
        public String toString() {
            return "ETagChangeDetectionPolicy mode=" + (Object)((Object)this.getMode());
        }
    }

    public static enum Mode {
        Client("client"),
        Server("server"),
        Warn("warn"),
        None("none");

        private final String mode;

        private Mode(String mode) {
            this.mode = mode;
        }

        private static Mode fromString(String trimmed) {
            for (Mode value : Mode.values()) {
                if (!value.mode.equals(trimmed)) continue;
                return value;
            }
            LOG.warn("Unrecognized fs.s3a.change.detection.mode value: \"{}\"", (Object)trimmed);
            return Mode.fromString("server");
        }

        static Mode fromConfiguration(Configuration configuration) {
            String trimmed = configuration.get("fs.s3a.change.detection.mode", "server").trim().toLowerCase(Locale.ENGLISH);
            return Mode.fromString(trimmed);
        }
    }

    public static enum Source {
        ETag("etag"),
        VersionId("versionid"),
        None("none");

        private final String source;

        private Source(String source) {
            this.source = source;
        }

        private static Source fromString(String trimmed) {
            for (Source value : Source.values()) {
                if (!value.source.equals(trimmed)) continue;
                return value;
            }
            LOG.warn("Unrecognized fs.s3a.change.detection.source value: \"{}\"", (Object)trimmed);
            return Source.fromString("etag");
        }

        static Source fromConfiguration(Configuration configuration) {
            String trimmed = configuration.get("fs.s3a.change.detection.source", "etag").trim().toLowerCase(Locale.ENGLISH);
            return Source.fromString(trimmed);
        }
    }
}

