# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
from __future__ import annotations

from typing import TYPE_CHECKING

from flask import request
from flask.json.tag import TaggedJSONSerializer
from flask.sessions import SecureCookieSessionInterface
from flask_session.sqlalchemy import SqlAlchemySessionInterface

if TYPE_CHECKING:
    from flask_session import Session


class SessionExemptMixin:
    """Exempt certain blueprints/paths from autogenerated sessions."""

    def save_session(self, *args, **kwargs):
        """Prevent creating session from REST API and health requests."""
        if request.blueprint == "/api/v1":
            return None
        if request.path == "/health":
            return None
        return super().save_session(*args, **kwargs)


class AirflowTaggedJSONSerializer(TaggedJSONSerializer):
    """
    Adapter of flask's serializer for flask-session.

    This serializer is used instead of MsgPackSerializer from flask-session,
    because MsgPackSerializer does not support Markup objects which need to be serialized in Airflow.

    So we are using TaggedJSONSerializer which is compatible with Markup objects but we pretend we have
    the same interface as MsgPackSerializer (encode/decode methods) to be compatible with flask-session.
    """

    def encode(self, session: Session) -> bytes:
        """Serialize the session data."""
        return self.dumps(session).encode()

    def decode(self, data: bytes) -> Session:
        """Deserialize the session data."""
        return self.loads(data.decode())


class AirflowDatabaseSessionInterface(SessionExemptMixin, SqlAlchemySessionInterface):
    """Session interface that exempts some routes and stores session data in the database."""

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.serializer = AirflowTaggedJSONSerializer()


class AirflowSecureCookieSessionInterface(SessionExemptMixin, SecureCookieSessionInterface):
    """Session interface that exempts some routes and stores session data in a signed cookie."""
